/*
 * Copyright 2002-2021 Dr. Jalal Kiswani. 
 * Email: Kiswanij@Gmail.com
 * Check out https://smart-api.com for more details
 * 
 * All the opensource projects of Dr. Jalal Kiswani are free for personal and academic use only, 
 * for commercial usage and support, please contact the author.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jk.webstack.controllers;

import java.util.List;
import java.util.Map;

import javax.faces.context.FacesContext;

import com.jk.core.config.JKConstants;
import com.jk.core.factory.JKFactory;
import com.jk.core.test.JKMockUtil;
import com.jk.core.util.JK;
import com.jk.core.util.JKCollectionUtil;
import com.jk.core.util.JKObjectUtil;
import com.jk.data.dataaccess.orm.JKObjectDataAccess;
import com.jk.data.dataaccess.orm.JKObjectDataAccessImpl;
import com.jk.services.client.workflow.JKWorkflowServiceClient;
import com.jk.services.client.workflow.JKWorkflowUtil;
import com.jk.services.client.workflow.JKWorkflowServiceClient.UserAction;
import com.jk.services.client.workflow.models.NewPayloadRequestModel;
import com.jk.services.client.workflow.models.PayloadModel;
import com.jk.services.client.workflow.models.WorkflowEntityModel;
import com.jk.web.faces.mb.JKManagedBean;
import com.jk.web.util.JKJsfUtil;
import com.jk.webstack.services.workflow.WorkflowController;

// TODO: Auto-generated Javadoc
/**
 * The Class JKManagedBeanWithOrmSupport.
 *
 * @param <T> the generic type
 */
public abstract class JKManagedBeanWithOrmSupport<T> extends JKManagedBeanWithSqlDataAccess {

	/** The model. */
	private T model;

	/** The oruginal model, which will be used in workflow */
	private T original;

	/** The model class. */
	private Class<T> modelClass;

	/** The model list. */
	private List<T> modelList;

	/** The filter list. */
	private List<T> filterList;

	/** In case required for another datatable. */
	private List filterList2;

	/** The always refresh list. */
	private boolean alwaysRefreshList;

	/** The mode. */
	protected ControllerMode mode;

	/** The confirm reeset. */
	private boolean confirmReeset;

	/** The edit tabular. */
	boolean editTabular;

	JKWorkflowServiceClient client = new JKWorkflowServiceClient();
	boolean workflowAvialable;
	PayloadModel payload;
	List<PayloadModel> payloads;

	private WorkflowEntityModel workflowEntity;

	/**
	 * Instantiates a new JK managed bean with orm support.
	 */
	public JKManagedBeanWithOrmSupport() {
		modelClass = JKFactory.type(JKObjectUtil.getGenericClassFromParent(this));
		mode = ControllerMode.ADD;
		workflowAvialable = JKWorkflowUtil.isWorkflowAvialable();
		if (workflowAvialable) {
			client.getSystem(JK.getAppName());// to sync App Name;
			workflowEntity = client.getWorkflowEntity(JK.getAppName(), modelClass.getSimpleName(),
					modelClass.getName());
		}
	}

	/**
	 * Adds the.
	 *
	 * @return the string
	 */
	public String addToDatabase() {
		if (model == null) {
			throw new IllegalStateException("Model is null while calling merge");
		}

		beforeInsert();
		T model = this.model;
		getDataAccess().insert(model);
		afterInsert();
		modelList = null;
		// reset();
		find(getIdValue(model));
		success("Added successfully", false);
		return null;
	}

	/**
	 * Edits the.
	 *
	 * @return the string
	 */
	public String edit() {
		mode = ControllerMode.EDIT;
		return null;

	}

	/**
	 * Save.
	 *
	 * @return the string
	 */
	public String saveToDatabase() {
		if (model == null) {
			throw new IllegalStateException("Model is null while calling merge");
		}
		beforeUpdate();
		T model = this.model;
		getDataAccess().update(model);
		reset();
		afterUpdate();
		find(getIdValue(model));
		success("Updated successfully", true);
		return null;
	}

	/**
	 * Before insert.
	 */
	protected void beforeInsert() {
	}

	/**
	 * After insert.
	 */
	protected void afterInsert() {
	}

	/**
	 * Before update.
	 */
	protected void beforeUpdate() {

	}

	/**
	 * After update.
	 */
	protected void afterUpdate() {

	}

	/**
	 * Before delete.
	 */
	protected void beforeDelete() {

	}

	/**
	 * Find.
	 *
	 * @param id the id
	 * @return the string
	 */
	public String find(int id) {
		T model = (T) getDataAccess().find(getModelClass(), id);
		setModel(model);
		mode = ControllerMode.READONLY;
		return null;
	}

	/**
	 * Gets the model class.
	 *
	 * @return the model class
	 */
	protected Class<T> getModelClass() {
		return modelClass;
	}

	/**
	 * Gets the model list.
	 *
	 * @return the model list
	 */
	public List<T> getModelList() {
		if (modelList == null) {
			modelList = getDataAccess().getList(getModelClass());
		}
		return modelList;
	}

	/**
	 * Delete.
	 *
	 * @return the string
	 */
	// ///////////////////////////////////////////////////
	public String deleteFromDatabase() {
		beforeDelete();
		getDataAccess().delete(getModelClass(), getIdValue());
		afterDelete();
		reset();
		success("Deleted successfully", false);
		return null;
	}

	/**
	 * After delete.
	 */
	protected void afterDelete() {
	}

	/**
	 * Gets the id value.
	 *
	 * @return the id value
	 */
	public Integer getIdValue() {
		return getIdValue(getModel());
	}

	/**
	 * Gets the id value.
	 *
	 * @param model the model
	 * @return the id value
	 */
	protected Integer getIdValue(T model) {
		return (Integer) JKObjectUtil.getFieldValue(model, JKConstants.Database.DEFAULT_PRIMARY_KEY_FIELD_NAME);
	}

	/**
	 * Gets the model.
	 *
	 * @return the model
	 */
	public T getModel() {
		if (model == null) {
			model = createEmptyModel();
		}
		return model;
	}

	/**
	 * Creates the empty model.
	 *
	 * @return the t
	 */
	protected T createEmptyModel() {
		return JKObjectUtil.newInstance(getModelClass());
	}

	/**
	 * Sets the model.
	 *
	 * @param model the new model
	 */
	public void setModel(T model) {
		// to avoid loosing any local views value
		if (model != this.model) {
			this.model = model;
			if (model == null) {
				mode = ControllerMode.ADD;
			} else {
				mode = ControllerMode.READONLY;
			}
		}
	}

	/**
	 * Gets the data access.
	 *
	 * @return the data access
	 */
	// ///////////////////////////////////////////////////
	protected JKObjectDataAccess getDataAccess() {
		return JKFactory.instance(JKObjectDataAccessImpl.class);
	}

	/**
	 * Reset.
	 *
	 * @return the string
	 */
	// ///////////////////////////////////////////////////
	public String reset() {
		this.model = null;
		this.modelList = null;
		payloads = null;
		original = null;
		mode = ControllerMode.ADD;
		if (isWorkflowAvialable()) {
			resetWorkFlow();
		}
		return null;
	}

	/**
	 * 
	 */
	protected void resetWorkFlow() {
		String controllerName = "workflow";
		WorkflowController workflow=getViewScopedManagedBean(controllerName);
				
		if (workflow != null) {
			workflow.reset();
			updateUi("frmNotifications");
		}
	}


	/**
	 * Checks if is always refresh list.
	 *
	 * @return true, if is always refresh list
	 */
	// ///////////////////////////////////////////////////
	public boolean isAlwaysRefreshList() {
		return alwaysRefreshList;
	}

	/**
	 * Sets the always refresh list.
	 *
	 * @param alwaysRefreshList the new always refresh list
	 */
	public void setAlwaysRefreshList(boolean alwaysRefreshList) {
		this.alwaysRefreshList = alwaysRefreshList;
	}

	/**
	 * Sets the id value.
	 *
	 * @param value the new id value
	 */
	public void setIdValue(Object value) {
		JKObjectUtil.setPeopertyValue(getModel(), JKConstants.Database.DEFAULT_PRIMARY_KEY_FIELD_NAME, value);
	}

	/**
	 * Duplicate.
	 *
	 * @return the string
	 */
	public String duplicate() {
		model = getDataAccess().clone(getModel());
		setIdValue(0);
		mode = ControllerMode.ADD;
		return null;
	}

	/**
	 * Gets the empty model.
	 *
	 * @return the empty model
	 */
	public T getEmptyModel() {
		return JKFactory.instance(modelClass);
	}

	/**
	 * Fill.
	 *
	 * @return the string
	 */
	public String fill() {
		JKMockUtil.fillFields(getModel());
		return null;
	}

	/**
	 * Checks if is allow add.
	 *
	 * @return true, if is allow add
	 */
	public boolean isAllowAdd() {
		return mode == ControllerMode.ADD;
	}

	/**
	 * Checks if is allow edit.
	 *
	 * @return true, if is allow edit
	 */
	public boolean isAllowEdit() {
		return mode == ControllerMode.READONLY;
	}

	/**
	 * Checks if is allow save.
	 *
	 * @return true, if is allow save
	 */
	public boolean isAllowSave() {
		return mode == ControllerMode.EDIT;
	}

	/**
	 * Checks if is allow delete.
	 *
	 * @return true, if is allow delete
	 */
	public boolean isAllowDelete() {
		return mode == ControllerMode.EDIT;
	}

	/**
	 * Checks if is allow reset.
	 *
	 * @return true, if is allow reset
	 */
	public boolean isAllowReset() {
		return getMode() != ControllerMode.WORKFLOW;
	}

	/**
	 * Checks if is allow fill.
	 *
	 * @return true, if is allow fill
	 */
	public boolean isAllowFill() {
		return isAllowAdd() && isDevelopmentMode();
	}

	/**
	 * Checks if is edits the mode.
	 *
	 * @return true, if is edits the mode
	 */
	public boolean isEditMode() {
		return mode == ControllerMode.EDIT;
	}

	/**
	 * Cancel edit.
	 */
	public void cancelEdit() {
		mode = ControllerMode.READONLY;
	}

	/**
	 * Checks if is read only mode.
	 *
	 * @return true, if is read only mode
	 */
	public boolean isReadOnlyMode() {
		return mode == ControllerMode.READONLY;
	}

	/**
	 * Gets the filter list.
	 *
	 * @return the filter list
	 */
	public List<T> getFilterList() {
		return filterList;
	}

	/**
	 * Sets the filter list 2.
	 *
	 * @param filterList2 the new filter list 2
	 */
	public void setFilterList2(List filterList2) {
		this.filterList2 = filterList2;
	}

	/**
	 * Sets the filter list.
	 *
	 * @param filterList the new filter list
	 */
	public void setFilterList(List<T> filterList) {
		this.filterList = filterList;
	}

	/**
	 * Gets the filter list 2.
	 *
	 * @return the filter list 2
	 */
	public List getFilterList2() {
		return filterList2;
	}

	/**
	 * Checks if is confirm reeset.
	 *
	 * @return true, if is confirm reeset
	 */
	public boolean isConfirmReeset() {
		return confirmReeset;
	}

	/**
	 * Sets the confirm reeset.
	 *
	 * @param confirmReeset the new confirm reeset
	 */
	public void setConfirmReeset(boolean confirmReeset) {
		this.confirmReeset = confirmReeset;
	}

	/**
	 * Gets the mode.
	 *
	 * @return the mode
	 */
	public ControllerMode getMode() {
		return mode;
	}

	/**
	 * Checks if is edits the tabular.
	 *
	 * @return true, if is edits the tabular
	 */
	public boolean isEditTabular() {
		return editTabular;
	}

	/**
	 * Sets the edits the tabular.
	 *
	 * @param editTabular the new edits the tabular
	 */
	public void setEditTabular(boolean editTabular) {
		this.editTabular = editTabular;
	}

	/**
	 * Save all.
	 */
	public void saveAllToDatabase() {
		List<T> list = getModelList();
		for (T model : list) {
			getDataAccess().update(model);
		}
		success("Saved succesfully", false);
	}

	/**
	 * 
	 * @return
	 */
	public String add() {
		if (isWorkflowAvialable()) {
			// @formatter:off
			client.insert(
				new NewPayloadRequestModel().
				withEntity(getWorkflowEntityName()).
				withAction(UserAction.CREATE.toString()).
				withBody(JKObjectUtil.toJson(getModel())).
				withSystem(JK.getAppName()));
			// @formatter:on
			reset();
			success("Record sent for the needed approvals");
		} else {
			return addToDatabase();
		}
		return null;
	}

	/**
	 * @return
	 */
	public String save() {
		if (isWorkflowAvialable()) {
			// @formatter:off
			client.insert(
				new NewPayloadRequestModel().
				withEntity(getWorkflowEntityName()).
				withAction(UserAction.MODIFY.toString()).
				withBody(JKObjectUtil.toJson(getModel())).
				withSystem(JK.getAppName()));
			// @formatter:on
			reset();
			success("Record sent for the needed approvals");
		} else {
			return saveToDatabase();
		}
		return null;
	}

	/**
	 * @return
	 */
	public String saveAll() {
		if (isWorkflowAvialable()) {
			List<T> list = getModelList();
			for (T model : list) {
			// @formatter:off
			client.insert(
				new NewPayloadRequestModel().
				withEntity(getWorkflowEntityName()).
				withAction(UserAction.MODIFY.toString()).
				withBody(JKObjectUtil.toJson(model)).
				withSystem(JK.getAppName()));
			// @formatter:on
			}
			reset();
			success("Record sent for the needed approvals");
		} else {
			saveAllToDatabase();
			setEditTabular(false);
		}
		return null;
	}

	/**
	 * @return
	 */
	public String delete() {
		if (isWorkflowAvialable()) {
			// @formatter:off
			client.insert(
				new NewPayloadRequestModel().
				withEntity(getWorkflowEntityName()).
				withAction(UserAction.DELETE.toString()).
				withBody(JKObjectUtil.toJson(getModel())).
				withSystem(JK.getAppName()));
			// @formatter:on
			reset();
			success("Record sent for the needed approvals");
		} else {
			return deleteFromDatabase();
		}
		return null;
	}

	/**
	 * 
	 * @return
	 */
	public List<PayloadModel> getPayloads() {
		if (payloads == null) {
			payloads = client.getPayLoads(JK.getAppName(), getWorkflowEntityName(),
					JKCollectionUtil.toString(getRoles(), false));
		}
		return payloads;
	}

	/**
	 * 
	 * @param payloads
	 */
	public void setPayloads(List<PayloadModel> payloads) {
		this.payloads = payloads;
	}

	/**
	 * 
	 * @return
	 */
	protected String getWorkflowEntityName() {
		return getModelClass().getSimpleName();
	}

	/**
	 * 
	 * @return
	 */
	public boolean isWorkflowAvialable() {
		return workflowAvialable && workflowEntity != null;
	}

	/**
	 * 
	 * @return
	 */
	public PayloadModel getPayload() {
		return payload;
	}

	/**
	 * 
	 * @param payload
	 */
	public void setPayload(PayloadModel payload) {
		this.payload = payload;
	}

	/**
	 * @param id
	 * @return
	 */
	public String approvePayload(int id) {
		PayloadModel payload = findPayload(id);
		if (payload == null) {
			JK.exception("Payload with id ({}) not found." + id);
		}
		String approveAction = client.approveWorkflowEntity(id);
		if (approveAction.equals("INSERT")) {
			setModel(JKObjectUtil.jsonToObject(payload.getPayload(), modelClass));
			addToDatabase();
		}
		if (approveAction.equals("UPDATE")) {
			setModel(JKObjectUtil.jsonToObject(payload.getPayload(), modelClass));
			saveToDatabase();
		}
		if (approveAction.equals("DELETE")) {
			setModel(JKObjectUtil.jsonToObject(payload.getPayload(), modelClass));
			deleteFromDatabase();
		}
		reset();
		return null;
	}

	public PayloadModel findPayload(int id) {
		for (PayloadModel payload : getPayloads()) {
			if (payload.getId().equals(id)) {
				return payload;
			}
		}
		return null;
	}

	public String rejectPayload(int id) {
		String rejectAction = client.rejectWorkflowEntity(id);
		reset();
		return null;
	}

	public String viewPayload(int id) {
		this.payload = findPayload(id);
		mode = ControllerMode.WORKFLOW;
		this.model = (T) payload.getObject();
		return null;
	}

	public boolean isInputDisabled() {
		return mode == ControllerMode.READONLY || mode == ControllerMode.WORKFLOW;
	}

	public T getOriginal() {
		if (getPayload() != null && getPayload().getObject() != null) {
			int index = getModelList().indexOf(getPayload().getObject());
			if (index != -1) {
				original = getModelList().get(index);
			}
		}
		return original;
	}

	public void setOriginal(T original) {
		this.original = original;
	}

	public boolean isFieldModified(String name) {
		if (getMode() != ControllerMode.ADD && getOriginal() != null) {
			Object newValue = JKObjectUtil.getFieldValue(getModel(), name);
			Object originalValue = JKObjectUtil.getFieldValue(getOriginal(), name);
			if (newValue != null) {
				return !newValue.equals(originalValue);
			}
			if (originalValue != null) {
				return !originalValue.equals(newValue);
			}
		}
		return false;
	}
}
