/*
 * Copyright 2002-2021 Dr. Jalal Kiswani. 
 * Email: Kiswanij@Gmail.com
 * Check out https://smart-api.com for more details
 * 
 * All the opensource projects of Dr. Jalal Kiswani are free for personal and academic use only, 
 * for commercial usage and support, please contact the author.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jk.webstack.security.services;

import org.junit.Assert;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;

import com.jk.core.factory.JKFactory;
import com.jk.core.logging.JKLogger;
import com.jk.core.logging.JKLoggerFactory;
import com.jk.core.security.JKPasswordUtil;
import com.jk.core.util.JKValidationUtil;
import com.jk.data.dataaccess.orm.JKObjectDataAccess;
import com.jk.data.dataaccess.orm.JKObjectDataAccessImpl;
import com.jk.data.exceptions.JKRecordNotFoundException;
import com.jk.web.util.JKWebUtil;
import com.jk.webstack.security.User;
import com.jk.webstack.security.UserRole;

// TODO: Auto-generated Javadoc
/**
 * The Class SecurityService.
 */
public class SecurityService implements UserDetailsService {

	/** The logger. */
	static JKLogger logger = JKLoggerFactory.getLogger(SecurityService.class);

	/** The data access service. */
	JKObjectDataAccess dataAccessService = JKFactory.instance(JKObjectDataAccessImpl.class);

	/** The initialized. */
	// Check me, I have currently prioirity of faster startup, so delayed init to first call of findUser
	boolean initialized = false;

	/**
	 * Instantiates a new security service.
	 */
	//////////////////////////////////////////////////////////////////
	public SecurityService() {
	}

	/**
	 * Inits the.
	 */
	protected void init() {
		// validate if no user configured, then add default user
		if (dataAccessService.getList(User.class).size() == 0) {
			UserRole role = new UserRole();
			role.setName(UserRole.ADMIN);
			dataAccessService.insert(role);
			createAccount("admin", "admin-first-name", "admin-last-name", "admin", role);
		}
	}

	/**
	 * Load user by username.
	 *
	 * @param username the username
	 * @return the user
	 * @throws UsernameNotFoundException the username not found exception
	 */
	//////////////////////////////////////////////////////////////////
	@Override
	public User loadUserByUsername(String username) throws UsernameNotFoundException {
		if (!initialized) {
			init();
			initialized = true;
		}
		User account = findAccountUsername(username);
		if (account == null) {
			throw new UsernameNotFoundException(username);
		}
		return account;
	}

	/**
	 * Creates the account.
	 *
	 * @param username  the username
	 * @param firstname the firstname
	 * @param lastname  the lastname
	 * @param password  the password
	 * @param role      the role
	 * @return the user
	 */
	//////////////////////////////////////////////////////////////////
	public User createAccount(String username, String firstname, String lastname, String password, UserRole role) {
		logger.debug("Create account with ({},{},{})", username, firstname, lastname);
		User account = findAccountUsername(username);
		if (account == null) {
			account = User.create().username(username).firstname(firstname).lastname(lastname);
			account.setTemp(password);
			account.setRole(role);
			processNewAccount(account);
		} else {
			account.setTemp(password);
			processCurrentAccount(account);
		}
		return account;
	}

	/**
	 * Process new account.
	 *
	 * @param account the account
	 */
	//////////////////////////////////////////////////////////////////
	protected void processNewAccount(User account) {
		logger.debug("Process new account ({})", account.getUsername());
		String password = account.getTemp();
		account.setPassword(JKWebUtil.encodePassword(password));
		dataAccessService.insert(account);
		logger.debug("Account created ({}) with password ({})", account.getUsername(), account.getPassword());
	}

	/**
	 * Process current account.
	 *
	 * @param account the account
	 */
	//////////////////////////////////////////////////////////////////
	protected void processCurrentAccount(User account) {
		resetAccount(account);
	}

	/**
	 * Reset account.
	 *
	 * @param username the username
	 * @return the user
	 */
	//////////////////////////////////////////////////////////////////
	public User resetAccount(String username) {
		User acconut = findAccountUsername(username);
		if (acconut != null) {
			resetAccount(acconut);
			return acconut;
		} else {
			throw new JKRecordNotFoundException();
		}
	}

	/**
	 * Reset account.
	 *
	 * @param account the account
	 */
	//////////////////////////////////////////////////////////////////
	protected void resetAccount(User account) {
		logger.debug("Reset account ({})", account.getUsername());
		String password = account.getTemp();
		if (JKValidationUtil.isEmpty(password)) {
			password = JKPasswordUtil.generateNumricPassword(10);
			account.setTemp(password);
		}
		account.setPassword(new BCryptPasswordEncoder().encode(password));
		dataAccessService.update(account);
	}

	/**
	 * Find account username.
	 *
	 * @param username the username
	 * @return the user
	 */
	//////////////////////////////////////////////////////////////////
	protected User findAccountUsername(String username) {
		return dataAccessService.findOneByFieldName(User.class, "username", username);
	}

	/**
	 * Change password.
	 *
	 * @param userName        the user name
	 * @param currentPassword the current password
	 * @param newPassword     the new password
	 */
	//////////////////////////////////////////////////////////////////
	public void changePassword(String userName, String currentPassword, String newPassword) {
		User account = findAccountUsername(userName);
		Assert.assertTrue(account != null);
		Assert.assertTrue(JKWebUtil.matchPasswords(currentPassword, account.getPassword()));
		account.setPassword(JKWebUtil.encodePassword(newPassword));
		dataAccessService.update(account);
	}
}
