/*
 * Copyright 2002-2021 Dr. Jalal Kiswani. 
 * Email: Kiswanij@Gmail.com
 * Check out https://smart-api.com for more details
 * 
 * All the opensource projects of Dr. Jalal Kiswani are free for personal and academic use only, 
 * for commercial usage and support, please contact the author.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jk.webstack.services.email;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.jk.core.factory.JKFactory;
import com.jk.core.logging.JKLogger;
import com.jk.core.logging.JKLoggerFactory;
import com.jk.core.mail.JKMailInfo;
import com.jk.core.mail.JKMailSender;
import com.jk.core.util.JKConversionUtil;
import com.jk.data.dataaccess.orm.JKObjectDataAccess;
import com.jk.data.dataaccess.orm.JKObjectDataAccessImpl;

// TODO: Auto-generated Javadoc
/**
 * The Class EmailService.
 */
public class EmailService {
	
	/** The pool. */
	static ExecutorService pool = Executors.newCachedThreadPool();
	
	/** The logger. */
	static JKLogger logger = JKLoggerFactory.getLogger(EmailService.class);
	
	/** The data access service. */
	JKObjectDataAccess dataAccessService = JKFactory.instance(JKObjectDataAccessImpl.class);
	
	/** The sender. */
	private JKMailSender sender;

	/**
	 * Adds the email to queue.
	 *
	 * @param to the to
	 * @param subject the subject
	 * @param emailContents the email contents
	 * @param callSend the call send
	 */
	//////////////////////////////////////////////////////////////////////
	public void addEmailToQueue(String to, String subject, String emailContents, boolean callSend) {
		logger.debug("Adding new email to the queue To({}) with subject ({})", to, subject);
		Email email = new Email();
		email.setTo(to);
		email.setSubject(subject);
		email.setBody(emailContents);
		dataAccessService.insert(email);
		if (callSend) {
			pool.execute(() -> {
				sendEmailsInQueue();
			});
		}
	}

	/**
	 * Send emails in queue.
	 */
	//////////////////////////////////////////////////////////////////////
	public void sendEmailsInQueue() {
		logger.debug("Sending emails in Queue");
		JKMailInfo info = new JKMailInfo();
		info.setUsername(System.getProperty("jk-email-user", ""));
		info.setPassword(System.getProperty("jk-email-password", ""));
		info.setRequiresAuthentication(true);
		info.setSecured(true);

		List<Email> emails = getEmailsInQueue();
		JKMailSender emailSender = getEmailSender();
		info.setHost(System.getProperty("jk-email-server", ""));
		info.setSmtpPort(JKConversionUtil.toInteger(System.getProperty("jk-email-port", "")));
		info.setFrom("");
		info.setBcc("");

		logger.debug("Total emails ({})", emails.size());
		for (Email email : emails) {
			info.setTo(email.getTo());
			info.setSubject(email.getSubject());
			info.setMsg(email.getBody());

			emailSender.send(info);
			dataAccessService.delete(email);
		}
	}

	/**
	 * Gets the emails in queue.
	 *
	 * @return the emails in queue
	 */
	//////////////////////////////////////////////////////////////////////
	public List<Email> getEmailsInQueue() {
		List<Email> emails = dataAccessService.getList(Email.class);
		return emails;
	}

	/**
	 * Gets the email sender.
	 *
	 * @return the email sender
	 */
	//////////////////////////////////////////////////////////////////////
	protected JKMailSender getEmailSender() {
		if (sender == null) {
			logger.debug("No Email Sender is provided, default will be used");
			sender = new JKMailSender();
		}
		return sender;
	}

	/**
	 * Sets the sender.
	 *
	 * @param sender the new sender
	 */
	//////////////////////////////////////////////////////////////////////
	public void setSender(JKMailSender sender) {
		this.sender = sender;
	}
	
	/**
	 * The main method.
	 *
	 * @param args the arguments
	 */
	//////////////////////////////////////////////////////////////////////
	public static void main(String[] args) {
		EmailService s=new EmailService();
		s.addEmailToQueue("jalal@nevada.unr.edu", "Hi", "Hi Jalal, This is Naser, I just want to say hi", 
				true);
	}
}
