package com.morphcloud.api;

import com.morphcloud.navigator.node.NodeKey;
import com.morphcloud.navigator.client.Client;
import com.morphcloud.navigator.client.ClientRequestBuilder;
import com.morphcloud.http.HttpBody;
import com.morphcloud.navigator.node.*;
import org.apache.hc.core5.http.NameValuePair;

import java.util.List;
import java.util.Optional;

class ApiBase<E> {
    private final Client client;
    private final Node<E> node;
    private final Class<E> cls;

    ApiBase(Client client, String path, Class<E> cls) {
        this.client = client;
        this.cls = cls;
        this.node = getNode(path);
        throwNodeError();
    }

    ApiBase(ApiBase<?> base, String path, Class<E> cls) {
        this.client = base.getClient();
        this.cls = cls;
        this.node = getNode(path);
        throwNodeError();
    }

    ApiBase(ApiBase<?> base, String path, Class<E> cls, NameValuePair queryParam) {
        this.client = base.getClient();
        this.cls = cls;
        this.node = getNode(path, queryParam);
        throwNodeError();
    }

    ApiBase(ApiBase<?> base, String path, Class<E> cls, List<NameValuePair> queryParams) {
        this.client = base.getClient();
        this.cls = cls;
        this.node = getNode(path, queryParams);
        throwNodeError();
    }

    private Node<E> getNode(String path) {
        try {
            return client.<E>request(path).addContext().build().get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Node<E> getNode(String path, NameValuePair queryParam) {
        try {
            return client.<E>request(path).addParam(queryParam).addContext().build().get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Node<E> getNode(String path, List<NameValuePair> queryParams) {
        try {
            ClientRequestBuilder<E> request = client.request(path);
            for (NameValuePair param : queryParams) request.addParam(param);
            return request.addContext().build().get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected <E> Node<E> getUpdate(Class<E> cls, List<NameValuePair> queryParams) {
        return new ApiBase<>(this, this.getPath(), cls, queryParams).node;
    }

    private void throwNodeError() {
        this.node.getError().ifPresent(error -> {
            throw new ApiValidationException(String.format("Navigator response error: %s last successful route: %s", error.getMessage(), error.getLastSuccessfulRoute()));
        });
    }

    protected Class<E> getCls() {
        return cls;
    }

    public Client getClient() {
        return client;
    }

    protected NodeData<E> getData() {
        return node.getData().setClass(cls);
    }

    protected <B extends HttpBody<B>> E post(B newEntity) {
        try {
            Node<E> node = this.client.<E>request(this.getPath()).build().post(newEntity);
            return node.getData().setClass(this.cls).getEntity().orElseThrow();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected <B extends HttpBody<B>> E patch(B update) {
        try {
            Node<E> node = this.client.<E>request(this.getPath()).build().patch(update);
            return node.getData().setClass(this.cls).getEntity().orElseThrow();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    protected Node<E> delete() {
        try {
            return this.client.<E>request(this.getPath()).build().delete();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public NodeControlResource getResource(NodeKey key) {
        return node.getControl().getResource().get(key)
        .map(resource -> {
            if (!resource.getAuth().getAuthorized()) throw new ApiAuthException(resource);
            return resource;
        }).orElseThrow(ApiException::new);
    }

    public Optional<NodeAction> getAction(NodeActionType type) {
        return node.getControl().getAction().get(type);
    }

    public NodeControlResource getAuthorizedResource(NodeKey key) {
        NodeControlResource resource = getResource(key);
        if (!resource.getAuth().getAuthorized()) throw new ApiValidationException(String.format("Not authorized to access resource: %s", resource.getLinkTo()));
        return resource;
    }

    protected String getPath() {
        return node.getPath();
    }

    protected void requireContext(NodeKey key) {
        if (!node.getContext().containsKey(key)) throw new ApiValidationException(String.format("%s context required for path %s", key.toString(), node.getPath()));
    }

    protected void requireResource(NodeKey key) {
        if (!node.getControl().getResource().containsKey(key)) throw new ApiValidationException(String.format("%s resource required for path %s", key.toString(), node.getPath()));
    }
}
