package com.morphcloud.api.auth;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.morphcloud.navigator.client.Client;
import com.morphcloud.http.HttpBody;
import com.morphcloud.navigator.node.Node;
import com.morphcloud.navigator.node.NodeKey;

import java.time.Instant;

abstract class ApiAuthBase<B extends HttpBody<B>> implements ApiAuth {

    private final Client client;
    private final String path;
    private final B authPayload;
    private DecodedJWT jwt;

    ApiAuthBase(Client client, NodeKey key, B authPayload) {
        this.client = client;
        this.path = String.format("/%s/%s", NodeKey.session, key);
        this.authPayload = authPayload;
        this.jwt = fetchJwt();
    }

    @Override
    public String getJwt() {
        try {
            if (isTokenLive()) return jwt.getToken();
            this.jwt = fetchJwt();
            return this.jwt.getToken();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getBearer() {
        return String.format("Bearer %s", getJwt());
    }

    @Override
    public String subGlobalId() {
        return this.jwt.getSubject();
    }

    private DecodedJWT fetchJwt() {
        Node<String> res = client.<String>request(path).build().post(authPayload);
        String jwtStr = res.getData().setClass(String.class).getEntity().orElseThrow(() -> new RuntimeException("Authentication failed"));
        return JWT.decode(jwtStr);
    }

    private boolean isTokenLive() {
        long tokenExpiry = jwt.getExpiresAtAsInstant().toEpochMilli();
        long now = Instant.now().toEpochMilli() - 5000;
        return tokenExpiry > now;
    }
}