package com.ncryptify;

/**
 * Created by James FitzGerald on 8/18/15.
 */

import com.fasterxml.jackson.databind.ObjectMapper;

import javax.crypto.Cipher;
import javax.crypto.ShortBufferException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.util.logging.Logger;
import de.undercouch.bson4jackson.BsonFactory;


/**
 * Provides access to the ncryptify.com service allowing developers to quickly and easily add data protection to their
 * apps, while giving data owners control over their secured data.
 */
public class Client {

    // todo: consider changing to an enum for the API
    // Constants used with hide and unhide format preserving encryption methods
    public static final String FPE_DIGIT = "digit";
    public static final String FPE_ALPHABET = "alphabet";
    public static final String FPE_ALPHANUMERIC = "alphanumeric";
    public static final String FPE_PRINTABLE = "printable";
    public static final String BLOB = "blob";

    private static final Logger LOGGER = Logger.getLogger(Client.class.getName());

    private static final String BLOB_CRYPTO_ALG = "AES/GCM/NoPadding";
    private static final int AAD_TLEN = 16;
    private static final int GCM_IV_LEN = 12;
    private static final int BLOB_CRYPTO_BLOCK_SIZE = 16;
    private static final int CRYPTO_HEADER_VERSION = 1;
    private static final int CHUNK_SIZE = 1024;
    private static final int MEMORY_MAP_SIZE_MINIMUM = 1024 * 1024; // Anything over 1 MB will use memory mapped writes

    private RestClient restClient;

    /**
     * Construct a client using the clientID and clientSecret. Not recommended for insecure end points, use the JWT based
     * constructor instead.
     *
     * @param clientId         : Application Id generated from the ncryptify.com development portal
     * @param clientSecret     : Application secret generated from the ncryptify.com development portal
     * @param issuer           : Your application name registered with the ncryptify.com development portal
     * @param expiresInMinutes : A JWT is created for use with the ncryptify.com service. It will expire in this many minutes
     * @param subject          : The applications User ID. It is up to the application developers to define what this is.
     */
    public Client(String clientId, String clientSecret, String issuer, Long expiresInMinutes, String subject) {
        // todo: Refactor API to not use Long for expiresInMinutes but a simpe int will suffice
        restClient = new RestClient(clientId, clientSecret, issuer, subject, expiresInMinutes.intValue());
    }

    /**
     * Construct a client using a prebuilt JWT. This is the preferred approach for most deployments as the client does
     * not need to know the clientSecret.
     * JWTs can be constructed on more secure end points with calls to createJwt and then delivered to the end points.
     * @param jwt       : Encoded jwt authorizing API access on behalf of a user. Typically generated elsewhere
     */
    public Client(String jwt) {
        restClient = new RestClient(jwt);
    }

    /**
     * Sets the Proxy
     * @param proxyURL Proxy URL
     * @throws IllegalArgumentException throws if the url is not formatted correctly or unsupported protocol is specified.
     */
    public static void setProxy(String proxyURL) throws IllegalArgumentException {
        RestClient.setProxy(proxyURL);
    }

    public static String createJwt(String clientId, String clientSecret, String issuer, Long expiresInMinutes,
                                   String subject) {
        return RestClient.constructJwt(clientId, clientSecret, issuer, subject, expiresInMinutes.intValue());
    }

    /**
     * Hides/Encrypts value preserving the format
     * There are currently limitations to the lengths supported for specific hints. They all have a minimum
     * length of 2. The maximums are as follows:
     * FPE_DIGIT 52; FPE_ALPHABET 32; FPE_ALPHANUMERIC 32; FPE_PRINTABLE 28
     *
     * @param value       : Value to encrypt
     * @param keyNameOrId : Key name or Key ID to use, it will be created if it does not exist
     * @param hint        : Format type, it can be FPE_DIGIT ("digit"), FPE_ALPHABET ("alphabet"),
     *                    FPE_ALPHANUMERIC ("alphanumeric"), or FPE_PRINTABLE ("printable")
     * @return format preserved cipher text
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public String hide(String value, String keyNameOrId, String hint) throws NCryptifyException {
		return hide(value, keyNameOrId, hint, "");
    }
	
    /**
     * Hides/Encrypts value preserving the format
     * There are currently limitations to the lengths supported for specific hints. They all have a minimum
     * length of 2. The maximums are as follows:
     * FPE_DIGIT 52; FPE_ALPHABET 32; FPE_ALPHANUMERIC 32; FPE_PRINTABLE 28
     *
     * @param value       : Value to encrypt
     * @param keyNameOrId : Key name or Key ID to use, it will be created if it does not exist
     * @param hint        : Format type, it can be FPE_DIGIT ("digit"), FPE_ALPHABET ("alphabet"),
     *                    FPE_ALPHANUMERIC ("alphanumeric"), or FPE_PRINTABLE ("printable")
     * @param tweak       : Arbitrary string to be combined with 'key' for hiding 'value'.
     * @return format preserved cipher text
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public String hide(String value, String keyNameOrId, String hint, String tweak) throws NCryptifyException {
        return fpe("hide", value, keyNameOrId, hint, tweak);
    }

    /**
     * Unhides/Decrypts value hidden by a call to hide
     *
     * @param value       : Value to decrypt
     * @param keyNameOrId : Key name or Key ID to use.
     * @param hint        : Format type, it can be FPE_DIGIT ("digit"), FPE_ALPHABET ("alphabet"),
     *                    FPE_ALPHANUMERIC ("alphanumeric"), or FPE_PRINTABLE ("printable")
     * @return plain text
     * @throws NCryptifyException          : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException        : the key cannot be found and createIfNotFound is false
     * @throws UnsupportedVersionException : The blob was encrypted with an incompatible client
     */
    public String unhide(String value, String keyNameOrId, String hint) throws NCryptifyException,
            UnsupportedVersionException, KeyNotFoundException {
		return unhide(value, keyNameOrId, hint, "");
    }
    /**
     * Unhides/Decrypts value hidden by a call to hide
     *
     * @param value       : Value to decrypt
     * @param keyNameOrId : Key name or Key ID to use.
     * @param hint        : Format type, it can be FPE_DIGIT ("digit"), FPE_ALPHABET ("alphabet"),
     *                    FPE_ALPHANUMERIC ("alphanumeric"), or FPE_PRINTABLE ("printable")
     * @param tweak       : String passed to the hide call.	
     * @return plain text
     * @throws NCryptifyException          : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException        : the key cannot be found and createIfNotFound is false
     * @throws UnsupportedVersionException : The blob was encrypted with an incompatible client
     */
    public String unhide(String value, String keyNameOrId, String hint, String tweak) throws NCryptifyException,
            UnsupportedVersionException, KeyNotFoundException {
        return fpe("unhide", value, keyNameOrId, hint, tweak);
    }

    /**
     * Encrypts the data from plainTextBuffer and writes it into cipherTextBuffer. Returns the number of bytes written.
     * cipherTextBuffer can be null in which case the number of bytes required is returned and the plainTextBuffer
     * position remains intact.
     *
     * @param plainTextBuffer  : ByteBuffer containing bytes to encrypt
     * @param cipherTextBuffer : ByteBuffer to write cipher text to. Can be null
     * @param keyNameOrId      : Key name or Key ID to use, it will be created if it does not exist
     * @return Number of bytes written into cipherTextBuffer or if cipherTextBuffer is null the number of bytes required
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public int encryptBlob(ByteBuffer plainTextBuffer, ByteBuffer cipherTextBuffer, String keyNameOrId)
            throws NCryptifyException {
        String errorMessage;

        // Encrypt the bytes
        //
        try {
            Key key = getKey(keyNameOrId);
            if (!key.getUsage().equals(BLOB)) {
            	throw new InvalidKeyTypeException("Invalid key type found: " + key.getUsage());
            }
            
            byte[] iv = getRandom(GCM_IV_LEN);
            byte[] header = createHeader(iv, key.getKeyId());

            Cipher aesCipher = Cipher.getInstance(BLOB_CRYPTO_ALG);
            aesCipher.init(Cipher.ENCRYPT_MODE, key.getSecretKey(), new GCMParameterSpec(AAD_TLEN * 8, iv));

            int outputSize = aesCipher.getOutputSize(plainTextBuffer.limit());

            // may just be a test of required size
            if (cipherTextBuffer != null) {
                // authenticate the header too
                //
                aesCipher.updateAAD(header);
                cipherTextBuffer.put(header);

                while (plainTextBuffer.remaining() > CHUNK_SIZE) {
                    // make a mini-me - the buffer contents are shared
                    ByteBuffer chunkBuffer = plainTextBuffer.slice();
                    chunkBuffer.limit(CHUNK_SIZE);
                    aesCipher.update(chunkBuffer, cipherTextBuffer);
                    // shift the position by a chunk
                    plainTextBuffer.position(CHUNK_SIZE + plainTextBuffer.position());
                }

                aesCipher.doFinal(plainTextBuffer, cipherTextBuffer);
                // note: Java puts the tag at the end of the cipher stream
            }

            return outputSize + header.length;

        } catch (java.security.NoSuchAlgorithmException e) {
            errorMessage = e.getMessage();
        } catch (java.security.InvalidKeyException e) {
            errorMessage = e.getMessage();
            if (errorMessage.contains("Illegal key size")) {
                errorMessage = errorMessage.concat(": Check that the Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy is configured (see the README.md)");
            }
        } catch (javax.crypto.IllegalBlockSizeException e) {
            errorMessage = e.getMessage();
        } catch (javax.crypto.NoSuchPaddingException e) {
            errorMessage = e.getMessage();
        } catch (java.security.InvalidAlgorithmParameterException e) {
            errorMessage = e.getMessage();
        } catch (javax.crypto.BadPaddingException e) {
            errorMessage = e.getMessage();
        } catch (javax.crypto.ShortBufferException e) {
            errorMessage = e.getMessage();
        } catch (IOException e) {
            errorMessage = e.getMessage();
        } catch (InvalidKeyTypeException e) {
        	errorMessage = e.getMessage();
        }
        LOGGER.warning(errorMessage);
        throw new NCryptifyException(errorMessage);
    }

    /**
     * Encrypts the data from plainTextBytes and returns the cipher text.
     *
     * @param plainTextBytes : byte array to encrypt
     * @param keyName        : Key name or Key ID to use, it will be created if it does not exist
     * @return encrypted bytes
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public byte[] encryptBlob(byte[] plainTextBytes, String keyName) throws NCryptifyException {
        ByteBuffer plainTextBuffer = ByteBuffer.wrap(plainTextBytes);
        ByteBuffer cipherTextBuffer = ByteBuffer.allocate(encryptBlob(plainTextBuffer, null, keyName));
        encryptBlob(plainTextBuffer, cipherTextBuffer, keyName);
        return cipherTextBuffer.array();
    }

    /**
     * Decrypts the data from cipherTextBuffer and writes it into plainTextBuffer. Returns the number of bytes written.
     * plainTextBuffer can be null in which case the number of bytes required is returned and cipherTextBuffer position
     * remains intact.
     *
     * @param cipherTextBuffer : ByteBuffer containing the bytes to decrypt
     * @param plainTextBuffer  : ByteBuffer to write the plain text into. Can be null
     * @return Number of bytes written into plainTextBuffer or if plainTextBuffer is null the number of bytes required
     * @throws NCryptifyException          : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException        : the key cannot be found and createIfNotFound is false
     * @throws UnsupportedVersionException : The blob was encrypted with an incompatible client
     */
    public int decryptBlob(ByteBuffer cipherTextBuffer, ByteBuffer plainTextBuffer) throws NCryptifyException,
            KeyNotFoundException, UnsupportedVersionException {
        String errorMessage;

        // Decrypt the bytes
        //
        try {
            int cipherTextBufferOriginalPosition = cipherTextBuffer.position();
            int headerLength = getHeaderLength(cipherTextBuffer);
            byte[] cipherTextHeaderBytes = new byte[headerLength];
            cipherTextBuffer.get(cipherTextHeaderBytes);

            CryptoHeader cryptoHeader = getHeader(cipherTextHeaderBytes);
            if (cryptoHeader.getVersion() > CRYPTO_HEADER_VERSION) {
                // A header version change indicates a non-backward compatible header change
                //
                throw new UnsupportedVersionException("Unsupported crypto header version");
            }
            // Look up the key
            //
            Key key = getKey(cryptoHeader.getKeyId(), false); 
            if (!key.getUsage().equals(BLOB)) {
            	throw new InvalidKeyTypeException("Invalid key type found: " + key.getUsage());
            }
            // Setup the cipher
            Cipher aesCipher = Cipher.getInstance(BLOB_CRYPTO_ALG);
            aesCipher.init(Cipher.DECRYPT_MODE, key.getSecretKey(), new GCMParameterSpec(AAD_TLEN * 8,
                    cryptoHeader.getIv()));

            int outputSize = aesCipher.getOutputSize(cipherTextBuffer.limit() - headerLength);
            if (plainTextBuffer == null) {
                cipherTextBuffer.position(cipherTextBufferOriginalPosition);
            } else {
                // authenticate the header too
                //
                aesCipher.updateAAD(cipherTextHeaderBytes);

                while (cipherTextBuffer.remaining() > CHUNK_SIZE) {
                    // make a mini-me - the buffer contents are shared
                    ByteBuffer chunkBuffer = cipherTextBuffer.slice();
                    chunkBuffer.limit(CHUNK_SIZE);
                    aesCipher.update(chunkBuffer, plainTextBuffer);
                    // shift the position by a chunk
                    cipherTextBuffer.position(CHUNK_SIZE + cipherTextBuffer.position());
                }

                aesCipher.doFinal(cipherTextBuffer, plainTextBuffer);
            }
            return outputSize;

        } catch (java.security.NoSuchAlgorithmException e) {
            errorMessage = e.getMessage();
        } catch (java.security.InvalidKeyException e) {
            errorMessage = e.getMessage();
            if (errorMessage.contains("Illegal key size")) {
                errorMessage = errorMessage.concat(": Check that the Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy is configured (see the README.md)");
            }
        } catch (javax.crypto.IllegalBlockSizeException e) {
            errorMessage = e.getMessage();
        } catch (javax.crypto.NoSuchPaddingException e) {
            errorMessage = e.getMessage();
        } catch (java.security.InvalidAlgorithmParameterException e) {
            errorMessage = e.getMessage();
        } catch (javax.crypto.BadPaddingException e) {
            errorMessage = e.getMessage();
        } catch (IOException e) {
            errorMessage = e.getMessage();
        } catch (BufferUnderflowException e) {
            errorMessage = e.getMessage();
        } catch (ShortBufferException e) {
            errorMessage = e.getMessage();
        } catch (InvalidKeyTypeException e) {
        	errorMessage = e.getMessage();
        }
        LOGGER.warning(errorMessage);
        throw new NCryptifyException(errorMessage);
    }

    /**
     * Decrypts cipherTextBytes and returns the plain text bytes
     *
     * @param cipherTextBytes : byte array to decrypt
     * @return decrypted bytes
     * @throws NCryptifyException          : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException        : the key cannot be found
     * @throws UnsupportedVersionException : The blob was encrypted with an incompatible client
     */
    public byte[] decryptBlob(byte[] cipherTextBytes) throws NCryptifyException, KeyNotFoundException,
            UnsupportedVersionException {
        ByteBuffer cipherTextBuffer = ByteBuffer.wrap(cipherTextBytes);
        ByteBuffer plainTextBuffer = ByteBuffer.allocate(decryptBlob(cipherTextBuffer, null));
        decryptBlob(cipherTextBuffer, plainTextBuffer);
        return plainTextBuffer.array();
    }

    /**
     * Encrypts plainTextFile writing into cipherTextFile
     *
     * @param plainTextFile  : File object to read data to encrypt from
     * @param cipherTextFile : File object to write cipher text to
     * @param keyNameOrId    : Key name or Key ID to use, it will be created if it does not exist
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public void encryptFile(File plainTextFile, File cipherTextFile, String keyNameOrId) throws NCryptifyException {
        try {
            processFile(plainTextFile, cipherTextFile, keyNameOrId, true);
        } catch (KeyNotFoundException e) {
            // should not happen!
            throw new NCryptifyException(e.getMessage());
        } catch (UnsupportedVersionException e) {
            // should not happen!
            throw new NCryptifyException(e.getMessage());
        }
    }

    /**
     * Encrypts plainTextFile writing into cipherTextFile using a default user key for file encryption operations
     *
     * @param plainTextFile  : File object to read data to encrypt from
     * @param cipherTextFile : File object to write cipher text to
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public void encryptFile(File plainTextFile, File cipherTextFile) throws NCryptifyException {
        encryptFile(plainTextFile, cipherTextFile, "defaultFileEncryptionKey");
    }

    private void processFile(File inputFile, File outputFile, String keyNameOrId, boolean encrypt)
            throws NCryptifyException, KeyNotFoundException, UnsupportedVersionException {
        String errorMessage;

        try {
            RandomAccessFile inputFileRA = new RandomAccessFile(inputFile, "r");
            boolean useMemoryMapping = inputFileRA.length() >= MEMORY_MAP_SIZE_MINIMUM;
            ByteBuffer inputFileBuffer = null;
            if (useMemoryMapping) {
                FileChannel inputFileChannel = inputFileRA.getChannel();
                MappedByteBuffer mappedInputFileBuffer = inputFileChannel.map(FileChannel.MapMode.READ_ONLY, 0,
                        inputFileChannel.size());
                mappedInputFileBuffer.load();
                inputFileBuffer = mappedInputFileBuffer;
            } else {
                inputFileBuffer = ByteBuffer.allocate((int) inputFileRA.length());
                inputFileBuffer.order(ByteOrder.LITTLE_ENDIAN);
                inputFileRA.readFully(inputFileBuffer.array());
            }

            int requiredSize = encrypt ? encryptBlob(inputFileBuffer, null, keyNameOrId) : decryptBlob(inputFileBuffer,
                    null);
            FileChannel outputFileChannel = new RandomAccessFile(outputFile, "rw").getChannel();

            if (useMemoryMapping) {
                MappedByteBuffer outputFileBuffer = outputFileChannel.map(FileChannel.MapMode.READ_WRITE, 0,
                        requiredSize);
                if (encrypt)
                    encryptBlob(inputFileBuffer, outputFileBuffer, keyNameOrId);
                else
                    decryptBlob(inputFileBuffer, outputFileBuffer);
            } else {

                ByteBuffer cipherTextFileBuffer = ByteBuffer.allocate(requiredSize);
                cipherTextFileBuffer.order(ByteOrder.LITTLE_ENDIAN);
                if (encrypt)
                    encryptBlob(inputFileBuffer, cipherTextFileBuffer, keyNameOrId);
                else
                    decryptBlob(inputFileBuffer, cipherTextFileBuffer);
                cipherTextFileBuffer.position(0);
                outputFileChannel.write(cipherTextFileBuffer, 0);
                outputFileChannel.close();
            }
            return;
        } catch (FileNotFoundException e) {
            errorMessage = e.getMessage();
        } catch (IOException e) {
            errorMessage = e.getMessage();
        }
        LOGGER.warning(errorMessage);
        throw new NCryptifyException(errorMessage);
    }

    /**
     * Decrypts cipherTextFile writing into plainTextFile
     *
     * @param cipherTextFile : File object to read cipher data from
     * @param plainTextFile  : File object to write plain text into
     * @throws NCryptifyException          : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException        : the key cannot be found
     * @throws UnsupportedVersionException : The blob was encrypted with an incompatible client
     */
    public void decryptFile(File cipherTextFile, File plainTextFile) throws NCryptifyException, KeyNotFoundException,
            UnsupportedVersionException {
        processFile(cipherTextFile, plainTextFile, null, false);
    }

    /**
     * Gets a user key identified by the given key name or id. It will create it if it does not exist
     *
     * @param keyNameOrId : Key name or Key ID to use, it will be created if it does not exist
     * @return The key
     * @throws NCryptifyException : if we get an unexpected error from ncryptify.com
     */
    public Key getKey(String keyNameOrId) throws NCryptifyException {
        try {
            return getKey(keyNameOrId, true);
        } catch (KeyNotFoundException e) {
            // should never happen!
            throw new NCryptifyException(e.getMessage());
        } 
    }

    /**
     * Gets a user key identified by the given key name or id. It will create it if it does not exist
     *
     * @param keyNameOrId      : Key name or Key ID to use, it will be created if it does not exist
     * @param createIfNotFound : Set to true if you want the key to be created if it does not exist
     * @return The key
     * @throws NCryptifyException      : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException    : the key cannot be found and createIfNotFound is false
     */
    public Key getKey(String keyNameOrId, boolean createIfNotFound) throws NCryptifyException, KeyNotFoundException {
        RestClient.KeyResponse keyResponse;
        try {
            keyResponse = restClient.getKey(keyNameOrId);
        } catch (KeyNotFoundException e) {
            if (!createIfNotFound) {
                throw e;
            }
            try {
                // We have to create it
                keyResponse = restClient.createKey(keyNameOrId);
            } catch (IOException ee) {
                String errorMessage = "Exception communicating with ncryptify: " + ee.getMessage();
                LOGGER.warning(errorMessage);
                throw new NCryptifyException(errorMessage);
            }
        } catch (IOException e) {
            String errorMessage = "Exception communicating with ncryptify: " + e.getMessage();
            LOGGER.warning(errorMessage);
            throw new NCryptifyException(errorMessage);
        }
        byte[] keyMaterial = hexToBytes(keyResponse.getMaterial());
        return new Key(new SecretKeySpec(keyMaterial, 0, keyMaterial.length, "AES"),
                keyResponse.getId(), keyResponse.getUsage());
    }

    public Account getAccount() throws NCryptifyException {
        try {
            return restClient.getAccount();
        } catch (IOException e) {
            String errorMessage = "Exception communicating with ncryptify: " + e.getMessage();
            LOGGER.warning(errorMessage);
            throw new NCryptifyException(errorMessage);
        }
    }

    /**
     * Deletes a user key identified by the given key name or id.
     *
     * @param keyNameOrId           : Key name or Key ID to use
     * @throws NCryptifyException   : if we get an unexpected error from ncryptify.com
     * @throws KeyNotFoundException : the key cannot be found
     */
    public void deleteKey(String keyNameOrId) throws NCryptifyException, KeyNotFoundException {
        try {
            restClient.deleteKey(keyNameOrId);
        } catch (IOException e) {
            String errorMessage = "Exception communicating with ncryptify: " + e.getMessage();
            LOGGER.warning(errorMessage);
            throw new NCryptifyException(errorMessage);
        }
    }

    /**
     * Gets the token (JWT) that the client is using. The creation and renewal of the token is handled internally.
     *
     * @return The token, renewed if necessary.
     */
    public String getToken() {
        return restClient.getToken();
    }


    //
    // Internals below....
    //

    /**
     * POJ for CryptoHeader construction
     */
    static class CryptoHeader {

        private int version;
        private String algorithm;
        private byte[] iv;
        private String keyId;

        public CryptoHeader() {
            this.version = CRYPTO_HEADER_VERSION;
        }

        public CryptoHeader(int version, String algorithm, byte[] iv, String keyId) {
            this.version = version;
            this.algorithm = algorithm;
            this.iv = iv;
            this.keyId = keyId;
        }

        public int getVersion() {
            return version;
        }

        public void setVersion(int version) {
            this.version = version;
        }

        public String getAlgorithm() {
            return algorithm;
        }

        public void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        public byte[] getIv() {
            return iv;
        }

        public void setIv(byte[] iv) {
            this.iv = iv;
        }

        public String getKeyId() {
            return keyId;
        }

        public void setKeyId(String keyId) {
            this.keyId = keyId;
        }
    }

    private byte[] createHeader(byte[] iv, String keyId) throws IOException {

        CryptoHeader cryptoHeader = new CryptoHeader(CRYPTO_HEADER_VERSION, "AES-GCM", iv, keyId);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectMapper mapper = new ObjectMapper(new BsonFactory());
        mapper.writeValue(baos, cryptoHeader);
        return baos.toByteArray();
    }

    private CryptoHeader getHeader(byte[] serializedHeader) throws IOException {
        ObjectMapper mapper = new ObjectMapper(new BsonFactory());
        return (CryptoHeader) mapper.readValue(serializedHeader, CryptoHeader.class);
    }

    private int getHeaderLength(ByteBuffer serializedHeaderBuffer) {
        serializedHeaderBuffer.order(ByteOrder.LITTLE_ENDIAN);
        return serializedHeaderBuffer.getInt(0);
    }

    private String fpe(String cmd, String value, String keyName, String hint, String tweak) throws NCryptifyException {
        if (value == null) {
            return null;
        }
        try {
            return restClient.postFpeAction(cmd, value, keyName, hint, tweak);
        } catch (IOException e) {
            String errorMessage = "Exception posting to crypto service :" + e.getMessage();
            LOGGER.warning(errorMessage);
            throw new NCryptifyException(errorMessage);
        }
    }

    private String fpeHide(String value, String keyName, String hint, String tweak) throws NCryptifyException {
        return fpe("hide", value, keyName, hint, tweak);
    }

    private String fpeUnhide(String value, String keyName, String hint, String tweak) throws NCryptifyException {
        return fpe("unhide", value, keyName, hint, tweak);
    }

    public byte[] getRandom(int size) throws NCryptifyException {
        try {
            return hexToBytes(restClient.getRandom(size).getMaterial());
        } catch (IOException e) {
            String errorMessage = "Exception retrieving random bytes: " + e.getMessage();
            LOGGER.warning(errorMessage);
            throw new NCryptifyException(errorMessage);
        }
    }

    // A minor utility function to create a HEX string.
    final protected static char[] hexArray = "0123456789ABCDEF".toCharArray();

    public static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = hexArray[v >>> 4];
            hexChars[j * 2 + 1] = hexArray[v & 0x0F];
        }
        return new String(hexChars);
    }

    public static byte[] hexToBytes(String hex) {
        int len = hex.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(hex.charAt(i), 16) << 4)
                    + Character.digit(hex.charAt(i + 1), 16));
        }
        return data;
    }

    private boolean isFpe(String hint) {
        return hint.equals(FPE_DIGIT) || hint.equals(FPE_ALPHABET) || hint.equals(FPE_ALPHANUMERIC)
                || hint.equals(FPE_PRINTABLE);
    }

}
