package com.ncryptify;

/**
 * Created by James FitzGerald on 9/15/15.
 */

import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpRequestFactory;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpResponseException;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.GenericJson;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.JsonObjectParser;
import com.google.api.client.json.jackson2.JacksonFactory;
import com.google.api.client.util.Key;
import com.google.api.client.http.HttpStatusCodes;
import com.google.api.client.http.json.JsonHttpContent;
import com.google.api.client.http.ByteArrayContent;

import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.Date;

/**
 * Rest API functionality used by Client class (you do not have to use this directly, everything you need is in the Client class)
 */
public class RestClient {

    private static HttpTransport HTTP_TRANSPORT;
    static final JsonFactory JSON_FACTORY = new JacksonFactory();
    public static final String NCRYPTIFY_URL;
    static {
        if (System.getenv("NCRYPTIFY_URL") != null) {
            NCRYPTIFY_URL = System.getenv("NCRYPTIFY_URL");
        } else {
            NCRYPTIFY_URL = "https://api.ncryptify.com/v1";
        }
    }

    private static String httpProxyHost;
    private static int httpProxyPort;

    private String clientId;
    private String clientSecret;
    private String issuer;
    private String subject;
    private int expiresInMinutes;
    private Date renewal;
    private String token;

    public RestClient(String clientId, String clientSecret, String issuer, String subject, int expiresInMinutes) {
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.issuer = issuer;
        this.subject = subject;
        this.expiresInMinutes = expiresInMinutes;
        this.renewal = setRenewal();
        this.token = constructJwt(clientId, clientSecret, issuer, subject, expiresInMinutes);
    }

    public RestClient(String jwt) {
        this.token = jwt;
    }

    /**
     * Sets the HTTP Proxy
     * @param proxyURL HTTP Proxy URL
     * @throws IllegalArgumentException throws if the url is not formatted correctly or non-http protocol is specified.
     */
    public static void setProxy(String proxyURL) throws IllegalArgumentException {
        if(proxyURL != null) {
            URL url;
            try {
                url = new URL(proxyURL);
            }
            catch (MalformedURLException e) {
                throw new IllegalArgumentException(e.getMessage());
            }
            if (!url.getProtocol().startsWith("http")) {
                throw new IllegalArgumentException("Protocol not supported");
            }
            httpProxyHost = url.getHost();
            httpProxyPort = url.getPort();
        }
        else {
            httpProxyHost = null;
            httpProxyPort = 0;
        }
        // NetHttpTransport does not handle shutdown()
        HTTP_TRANSPORT = null;
    }

    HttpTransport getHttpTransport() {
        if (HTTP_TRANSPORT == null) {
            if (httpProxyHost != null && !httpProxyHost.isEmpty()) {
                try {
                    Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(httpProxyHost, httpProxyPort));
                    HTTP_TRANSPORT = new NetHttpTransport.Builder().setProxy(proxy).build();
                } catch (Exception e) {
                    throw new ExceptionInInitializerError(e);
                }
            }
            else {
                HTTP_TRANSPORT = new NetHttpTransport.Builder().build();
            }
        }
        return HTTP_TRANSPORT;
    }

    public static class KeyResponse {

        @Key
        private String id;

        @Key
        private String name;

        // todo: meta
        //

        @Key
        private String material;

        @Key
        private String createdAt;

        @Key
        private String updatedAt;

        @Key
        private String deletedAt;
        
        @Key
        private String usage;

        public String getId() {
            return id;
        }

        public String getName() {
            return name;
        }

        public String getMaterial() {
            return material;
        }

        public String getCreatedAt() {
            return createdAt;
        }

        public String getUpdatedAt() {
            return updatedAt;
        }

        public String getDeletedAt() {
            return deletedAt;
        }
        
        public String getUsage() {
        	return usage;
        }
    }

    public static class LinkResponse {
        @Key("linkauth_url")
        private String linkAuthUrl;

        public String getLinkAuthUrl() {
            return linkAuthUrl;
        }
    }

    public static class RandomResponse {
        @Key
        private String material;

        public String getMaterial() {
            return material;
        }
    }

    public KeyResponse getKey(String keyNameOrId) throws IOException, KeyNotFoundException {
        HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) {
                request.getHeaders().setAuthorization("Bearer " + getToken());
                request.getHeaders().setUserAgent("ncryptify_client/java");
                request.getHeaders().setAccept("application/json");
                request.setParser(new JsonObjectParser(JSON_FACTORY));
            }
        });

        String name = URLEncoder.encode(keyNameOrId, "UTF-8");
        GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/vault/keys/" + name);
        HttpRequest request = requestFactory.buildGetRequest(url);
        try {
            HttpResponse response = request.execute();
            KeyResponse keyResponse = response.parseAs(KeyResponse.class);
            return keyResponse;
        } catch (HttpResponseException e) {
            if (e.getStatusCode() == HttpStatusCodes.STATUS_CODE_NOT_FOUND) {
                throw new KeyNotFoundException("Key not found");
            } else {
                throw e;
            }
        }
    }

    public KeyResponse createKey(String keyNameOrId) throws IOException {

        HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) {
                request.getHeaders().setAuthorization("Bearer " + getToken());
                request.getHeaders().setUserAgent("ncryptify_client/java");
                request.getHeaders().setContentType("application/json");
                request.getHeaders().setAccept("application/json");
                request.setParser(new JsonObjectParser(JSON_FACTORY));
            }
        });

        KeyRequestBody keyRequestBody = new KeyRequestBody(keyNameOrId);
        // Clients can only create keys for blob encryption.
        GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/vault/keys?returnExisting=true");
        HttpRequest request = requestFactory.buildPostRequest(url, new JsonHttpContent(JSON_FACTORY, keyRequestBody));
        return request.execute().parseAs(KeyResponse.class);
    }

    public void deleteKey(String keyNameOrId) throws IOException, KeyNotFoundException {
        try {
            HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
                @Override
                public void initialize(HttpRequest request) {
                    request.getHeaders().setAuthorization("Bearer " + getToken());
                    request.getHeaders().setUserAgent("ncryptify_client/java");
                    request.getHeaders().setAccept("application/json");
                    request.setParser(new JsonObjectParser(JSON_FACTORY));
                }
            });

            GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/vault/keys/" + keyNameOrId);
            HttpRequest request = requestFactory.buildDeleteRequest(url);
            request.execute();

        } catch (HttpResponseException e) {
            // check if its a 404
            if (e.getStatusCode() == HttpStatusCodes.STATUS_CODE_NOT_FOUND) {
                throw new KeyNotFoundException("Key not found");
            } else {
                throw e;
            }
        }
    }

    public RandomResponse getRandom(int bytes) throws IOException {
        HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) {
                request.getHeaders().setAuthorization("Bearer " + getToken());
                request.getHeaders().setUserAgent("ncryptify_client/java");
                request.getHeaders().setAccept("application/json");
                request.setParser(new JsonObjectParser(JSON_FACTORY));
            }
        });
        GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/vault/random?bytes=" + Integer.toString(bytes));
        HttpRequest request = requestFactory.buildGetRequest(url);
        HttpResponse response = request.execute();
        RandomResponse randomResponse = response.parseAs(RandomResponse.class);
        return randomResponse;
    }

    public String postFpeAction(String cmd, String value, String keyName, String hint) throws IOException {
		return postFpeAction(cmd, value, keyName, hint, "");
    }
	
    public String postFpeAction(String cmd, String value, String keyName, String hint, String tweak) throws IOException {
        HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) {
                request.getHeaders().setAuthorization("Bearer " + getToken());
                request.getHeaders().setUserAgent("ncryptify_client/java");
                request.getHeaders().setContentType("text/plain;");
                request.getHeaders().setAccept("text/plain");
            }
        });
        GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/crypto/" + cmd)
                .set("keyName", keyName)
                .set("hint", hint);
		if (tweak.length() > 0) {
			url.set("tweak", tweak);
		}
        // todo: is there something that is a better match for a StringContent
        HttpRequest request = requestFactory.buildPostRequest(url, new ByteArrayContent("text",
                value.getBytes(Charset.forName("UTF-8"))));
        HttpResponse response = request.execute();
        return response.parseAsString();
    }

    public Account getAccount() throws IOException {
        HttpRequestFactory requestFactory = getHttpTransport().createRequestFactory(new HttpRequestInitializer() {
            @Override
            public void initialize(HttpRequest request) {
                request.getHeaders().setAuthorization("Bearer " + getToken());
                request.getHeaders().setUserAgent("ncryptify_client/java");
                request.getHeaders().setAccept("application/json");
                request.setParser(new JsonObjectParser(JSON_FACTORY));
            }
        });

        GenericUrl url = new GenericUrl(NCRYPTIFY_URL + "/admin/account");
        HttpRequest request = requestFactory.buildGetRequest(url);
        return request.execute().parseAs(Account.class);
    }

    public String getToken() {
        Date now = new Date();
        // we only construct a JWT if we have the clientId && secret
        if (this.clientId != null && this.clientSecret != null && now.after(this.renewal)) {
            this.renewal = setRenewal();
            this.token = constructJwt(this.clientId, this.clientSecret, this.issuer, this.subject,
                    this.expiresInMinutes);
        }
        return this.token;
    }
    
    KeyRequestBody getKeyRequestBody(String name, String meta, String usage) {
    	return new KeyRequestBody(name,  meta,  usage);
    }

    public static class KeyRequestBody extends GenericJson {
        @Key("name")
        private String name;

        @Key
        private String meta;
        
        @Key
        private String usage;

        KeyRequestBody(String name) {
            this.name = name;
        }

        KeyRequestBody(String name, String meta, String usage) {
            this.name = name;
            this.meta = meta;
            this.usage = usage;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }
        
        public String getUsage() {
        	return usage;
        }
        
        public void setUsage(String usage) {
        	this.usage = usage;
        }

        public String getMeta() {
            return meta;
        }

        public void setMeta(String meta) {
            this.meta = meta;
        }
    }

    public static class LinkRequestBody extends GenericJson {

        LinkRequestBody(String callbackUrl) {
            this.callback = callbackUrl;
        }

        public String getCallback() {
            return callback;
        }

        @Key
        private String callback;
    }

    public static String constructJwt(String clientId, String clientSecret, String issuer, String subject,
                                int expiresInMinutes) {
        JwtBuilder jwtBuilder = Jwts.builder();
        jwtBuilder.setHeaderParam("typ", "JWT");
        jwtBuilder.setAudience(clientId);
        jwtBuilder.setIssuer(issuer);
        jwtBuilder.setSubject(subject);
        Date now = new Date();
        jwtBuilder.setIssuedAt(now);
        jwtBuilder.setExpiration(new Date(now.getTime() + expiresInMinutes * 60 * 1000));
        return jwtBuilder.signWith(SignatureAlgorithm.HS256, clientSecret.getBytes(Charset.forName("UTF-8"))).compact();
    }

    private Date setRenewal() {
        Date now = new Date();
        // give 30 seconds grace
        Date when = new Date(now.getTime() + this.expiresInMinutes * 60 * 1000 - 30000);
        return when;
    }
}
