/*
 * Copyright (c) 2017, Numdata BV, The Netherlands.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Numdata nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NUMDATA BV BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.numdata.oss.db;

import java.util.regex.*;

import org.jetbrains.annotations.*;

/**
 * Identifies a database by its name and, optionally, a server host name and
 * port number. Useful for command-line tools where a full JDBC URL would result
 * in excessive typing.
 *
 * @author G. Meinders
 */
public class DatabaseName
{
	/**
	 * Database name.
	 */
	@Nullable
	private String _host;

	/**
	 * Host name.
	 */
	@Nullable
	private String _database;

	/**
	 * Port number.
	 */
	@Nullable
	private Integer _port;

	/**
	 * Constructs a new instance.
	 *
	 * @param database Database name.
	 */
	public DatabaseName( @Nullable final String database )
	{
		this( database, null, null );
	}

	/**
	 * Constructs a new instance.
	 *
	 * @param database Database name.
	 * @param host     Host name.
	 */
	public DatabaseName( @Nullable final String database, @Nullable final String host )
	{
		this( database, host, null );
	}

	/**
	 * Constructs a new instance.
	 *
	 * @param database Database name.
	 * @param host     Host name.
	 * @param port     Port number.
	 */
	public DatabaseName( @Nullable final String database, @Nullable final String host, @Nullable final Integer port )
	{
		_database = database;
		_host = host;
		_port = port;
	}

	/**
	 * Returns the database name.
	 *
	 * @return Database name.
	 */
	@Nullable
	public String getDatabase()
	{
		return _database;
	}

	/**
	 * Returns the host name.
	 *
	 * @return Host name.
	 */
	@Nullable
	public String getHost()
	{
		return _host;
	}

	/**
	 * Returns the port number.
	 *
	 * @return Port number.
	 */
	@Nullable
	public Integer getPort()
	{
		return _port;
	}

	@Override
	public String toString()
	{
		final String result;

		if ( _host == null )
		{
			result = _database;
		}
		else if ( _port == null )
		{
			result = _database + "@" + _host;
		}
		else
		{
			result = _database + "@" + _host + ":" + _port;
		}

		return result;
	}

	/**
	 * Returns a database name object with the value represented by the given
	 * string. The syntax is {@code database-name ("@" host-name (":"
	 * port-number)? )?}.
	 *
	 * @param string String to be parsed.
	 *
	 * @return Database name.
	 *
	 * @throws IllegalArgumentException if the given string can't be parsed.
	 */
	@NotNull
	public static DatabaseName valueOf( @NotNull final String string )
	{
		final Pattern pattern = Pattern.compile( "(\\w+)(?:@([\\w\\.]+)(?::([0-9]+))?)?" );
		final Matcher matcher = pattern.matcher( string );
		if ( !matcher.matches() )
		{
			throw new IllegalArgumentException( string );
		}

		final String database = matcher.group( 1 );
		final String host = matcher.group( 2 );
		final String port = matcher.group( 3 );

		return new DatabaseName( database, host, ( port == null ) ? null : Integer.valueOf( port ) );
	}

	/**
	 * Returns a database name object with the value represented by the given
	 * string. The given default values are used for any optional parts of the
	 * syntax that are not specified.
	 *
	 * @param string       String to be parsed.
	 * @param defaultValue Default values for optional fragments.
	 *
	 * @return Database name.
	 *
	 * @throws IllegalArgumentException if the given string can't be parsed.
	 */
	@NotNull
	public static DatabaseName valueOf( @NotNull final String string, @NotNull final DatabaseName defaultValue )
	{
		final DatabaseName result = valueOf( string );

		if ( result._host == null )
		{
			result._host = defaultValue.getHost();
		}

		if ( result._port == null )
		{
			result._port = defaultValue.getPort();
		}

		return result;
	}
}
