/*
 * Copyright (c) 2017, Numdata BV, The Netherlands.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Numdata nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NUMDATA BV BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.numdata.oss.template;

import java.io.*;

import org.jetbrains.annotations.*;

/**
 * Output generated by a {@link Template}.
 *
 * @author G. Meinders
 */
public class TemplateOutput
	implements Appendable
{
	/**
	 * Template to create output for.
	 */
	@NotNull
	private Template _context;

	/**
	 * Character sequence to append template output to.
	 */
	@NotNull
	private final Appendable _out;

	/**
	 * Current position in the output.
	 */
	private int _position;

	/**
	 * Cursor position.
	 */
	private int _cursorPosition;

	/**
	 * Constructs a new instance.
	 *
	 * @param   context     Template to create output for.
	 * @param   out         Character sequence to append template output to.
	 */
	public TemplateOutput( final Template context, final Appendable out )
	{
		_context = context;
		_out = out;
		_position = 0;
		_cursorPosition = 0;
	}

	/**
	 * Returns the template to create output for.
	 *
	 * @return  Template.
	 */
	@NotNull
	public Template getContext()
	{
		return _context;
	}

	/**
	 * Sets the cursor position to the current position in the output.
	 */
	public void setCursorPosition()
	{
		_cursorPosition = _position;
	}

	/**
	 * Returns the cursor position, relative to the start of the output.
	 *
	 * @return  Cursor position.
	 */
	public int getCursorPosition()
	{
		return _cursorPosition;
	}

	@Override
	public Appendable append( final char c )
		throws IOException
	{
		_position++;
		return _out.append( c );
	}

	@Override
	public Appendable append( final CharSequence csq, final int start, final int end )
		throws IOException
	{
		_position += end - start;
		return _out.append( csq, start, end );
	}

	@Override
	public Appendable append( final CharSequence csq )
		throws IOException
	{
		_position += csq.length();
		return _out.append( csq );
	}
}
