/*
 * Copyright 2017 @objectsql.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.objectsql.query;

import com.objectsql.spring.SpringUtils;
import com.objectsql.support.*;
import com.objectsql.annotation.RdTable;
import com.objectsql.utils.ORMUtils;

import java.util.*;

public class QueryImpl extends AbstractQueryImpl implements Query {

    private Class<?> table;
    private Class<?> returnClass;
    private List<Column> returnColumns = new ArrayList<Column>();
    private List<Column> fixedReturnColumns = new ArrayList<Column>();

    private List<Column> groups = new ArrayList<Column>();
    private List<Column> groupCountsSelectColumns = new ArrayList<Column>();

    public Query orderDesc(String name){
		orders.add(new Order(new Column(name), Order.DESC));
		return this;
	}

    public Query whereIsNull(String name){
        addCondition(new Condition().and(new Expression(new Column(name), ExpressionType.CDT_IS_NULL)));
        return this;
    }

    public Query whereIsNotNull(String name){
        addCondition(new Condition().and(new Expression(new Column(name), ExpressionType.CDT_IS_NOT_NULL)));
        return this;
    }

    public Query whereIsEmpty(String name){
        addCondition(new Condition().and(new Expression(new Column(name), ExpressionType.CDT_IS_EMPTY)));
        return this;
    }

    public Query whereIsNotEmpty(String name){
        addCondition(new Condition().and(new Expression(new Column(name), ExpressionType.CDT_IS_NOT_EMPTY)));
        return this;
    }

    @Override
    public <T,R> Query where(LambdaQuery<T,R> fieldFunction, ExpressionType type){
        return where(fieldFunction.getColumnName(), type);
    }

    @Override
    public Query where(String name, ExpressionType type){
        if(ExpressionType.CDT_IS_NULL == type
                || ExpressionType.CDT_IS_NOT_NULL == type
                || ExpressionType.CDT_IS_EMPTY == type
                || ExpressionType.CDT_IS_NOT_EMPTY == type
                ) {
            addCondition(new Condition().and(new Expression(new Column(name), type)));
        }
        return this;
    }

    @Override
    public <T,R> Query whereEqual(LambdaQuery<T,R> fieldFunction, Object value){
        return whereEqual(fieldFunction.getColumnName(), value);
    }

    @Override
    public Query whereEqual(String name, Object value) {
        return where(name, value, ExpressionType.CDT_EQUAL);
    }

    @Override
    public <T,R> Query whereNotEqual(LambdaQuery<T,R> fieldFunction, Object value){
        return whereNotEqual(fieldFunction.getColumnName(), value);
    }
    @Override
    public Query whereNotEqual(String name, Object value) {
        return where(name, value, ExpressionType.CDT_NOT_EQUAL);
    }

    @Override
    public Query whereLike(String name, String value) {
        return where(name, value, ExpressionType.CDT_LIKE);
    }

    @Override
    public Query whereNotLike(String name, String value) {
        return where(name, value, ExpressionType.CDT_NOT_LIKE);
    }

    @Override
    public Query whereStartWith(String name, String value) {
        return where(name, value, ExpressionType.CDT_START_WITH);
    }

    @Override
    public Query whereEndWith(String name, String value) {
        return where(name, value, ExpressionType.CDT_END_WITH);
    }

    @Override
    public Query whereNotStartWith(String name, String value) {
        return where(name, value, ExpressionType.CDT_NOT_START_WITH);
    }

    @Override
    public Query whereNotEndWith(String name, String value) {
        return where(name, value, ExpressionType.CDT_NOT_END_WITH);
    }

    @Override
    public <T,R> Query whereLike(LambdaQuery<T,R> fieldFunction, String value){
        return whereLike(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereNotLike(LambdaQuery<T,R> fieldFunction, String value){
        return whereNotLike(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereStartWith(LambdaQuery<T,R> fieldFunction, String value){
        return whereStartWith(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereEndWith(LambdaQuery<T,R> fieldFunction, String value){
        return whereEndWith(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereNotStartWith(LambdaQuery<T,R> fieldFunction, String value){
        return whereNotStartWith(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereNotEndWith(LambdaQuery<T,R> fieldFunction, String value){
        return whereNotEndWith(fieldFunction.getColumnName(), value);
    }

    @Override
    public Query whereLess(String name, Object value) {
        return where(name, value, ExpressionType.CDT_LESS);
    }

    @Override
    public Query whereLessEqual(String name, Object value) {
        return where(name, value, ExpressionType.CDT_LESS_EQUAL);
    }

    @Override
    public Query whereMore(String name, Object value) {
        return where(name, value, ExpressionType.CDT_MORE);
    }

    @Override
    public Query whereMoreEqual(String name, Object value) {
        return where(name, value, ExpressionType.CDT_MORE_EQUAL);
    }

    @Override
    public <T,R> Query whereLess(LambdaQuery<T,R> fieldFunction, Object value){
        return whereLess(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereLessEqual(LambdaQuery<T,R> fieldFunction, Object value){
        return whereLessEqual(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereMore(LambdaQuery<T,R> fieldFunction, Object value){
        return whereMore(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereMoreEqual(LambdaQuery<T,R> fieldFunction, Object value){
        return whereMoreEqual(fieldFunction.getColumnName(), value);
    }

    @Override
    public Query whereIn(String name, Collection value) {
        return where(name, value, ExpressionType.CDT_IN);
    }

    @Override
    public Query whereNotIn(String name, Collection value) {
        return where(name, value, ExpressionType.CDT_NOT_IN);
    }

    @Override
    public Query whereInValues(String name, Object... values) {
        List<Object> temp = null;
        if(values != null){
            temp = Arrays.asList(values);
        }else{
            temp = new ArrayList<Object>();
        }
        return where(name, temp, ExpressionType.CDT_IN);
    }

    @Override
    public Query whereNotInValues(String name, Object... values) {
        List<Object> temp = null;
        if(values != null){
            temp = Arrays.asList(values);
        }else{
            temp = new ArrayList<Object>();
        }
        return where(name, temp, ExpressionType.CDT_NOT_IN);
    }

    @Override
    public <T,R> Query whereIn(LambdaQuery<T,R> fieldFunction, Collection value){
        return whereIn(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereNotIn(LambdaQuery<T,R> fieldFunction, Collection value){
        return whereNotIn(fieldFunction.getColumnName(), value);
    }
    @Override
    public <T,R> Query whereInValues(LambdaQuery<T,R> fieldFunction, Object ... values){
        return whereInValues(fieldFunction.getColumnName(), values);
    }
    @Override
    public <T,R> Query whereNotInValues(LambdaQuery<T,R> fieldFunction, Object ... values){
        return whereNotInValues(fieldFunction.getColumnName(), values);
    }
    @Override
    public Query where(String name, Object value, ExpressionType type){
        addCondition(new Condition().and(new Expression(new Column(name), value, type)));
		return this;
	}

    @Override
    public <T,R> Query whereIsNull(LambdaQuery<T,R> fieldFunction){
        return whereIsNull(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query whereIsNotNull(LambdaQuery<T,R> fieldFunction){
        return whereIsNotNull(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query whereIsEmpty(LambdaQuery<T,R> fieldFunction){
        return whereIsEmpty(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query whereIsNotEmpty(LambdaQuery<T,R> fieldFunction){
        return whereIsNotEmpty(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query where(LambdaQuery<T,R> fieldFunction, Object value, ExpressionType type){
        return where(fieldFunction.getColumnName(), value, type);
    }

	public Query where(Condition condition) {
        if(condition != null) {
            addCondition(condition);
        }
		return this;
	}

    @Override
    public Query whereBetween(String name, Object value, Object andValue){
        if(!ORMUtils.isEmpty(value) && !ORMUtils.isEmpty(andValue)){
            Expression expression = new Expression(new Column(name), value, ExpressionType.CDT_BETWEEN);
            expression.setAndValue(andValue);
            addCondition(new Condition().and(expression));
        }
        return this;
    }

    @Override
    public <T,R> Query whereBetween(LambdaQuery<T,R> fieldFunction, Object value, Object andValue){
        return whereBetween(fieldFunction.getColumnName(), value, andValue);
    }

    @Override
    public Query where(Expression... expressions) {
        addCondition(new Condition().and(expressions));
        return this;
    }

    public Query group(String name) {
		groups.add(new Column(name));
		return this;
	}

    public List<Column> getGroupCountSelectColumns(){
        return groupCountsSelectColumns;
    }

    public Query groupCountSelectColumn(String name) {
        groupCountsSelectColumns.add(new Column(name));
        return this;
    }

	@Override
	public Query having(String name, Object value, ExpressionType type) {
        if(value == null){
            return this;
        }
        if("".equals(value)){
            return this;
        }
        addHaving(new Condition().or(new Expression(new Column(name), value, type)));

		return this;
	}

    @Override
    public <T,R> Query group(LambdaQuery<T,R> fieldFunction){
        return group(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query groupCountSelectColumn(LambdaQuery<T,R> fieldFunction){
        return groupCountSelectColumn(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query having(LambdaQuery<T,R> fieldFunction, Object value, ExpressionType type){
        return having(fieldFunction.getColumnName(), value, type);
    }
    @Override
	public Query having(Condition condition) {
        addCondition(condition);
		return this;
	}

    @Override
	public Query orderAsc(String name){
		orders.add(new Order(new Column(name), Order.ASC));
		return this;
	}

    @Override
    public <T,R> Query orderDesc(LambdaQuery<T,R> fieldFunction){
        return orderDesc(fieldFunction.getColumnName());
    }
    @Override
    public <T,R> Query orderAsc(LambdaQuery<T,R> fieldFunction){
        return orderAsc(fieldFunction.getColumnName());
    }


    public Query order(Order order) {
        if (order != null) {
            orders.add(order);
        }
        return this;
    }

    public Query orders(List<Order> os){
        if(os != null){
            for(Order order : os){
                orders.add(order);
            }
        }
        return this;
    }

    @Override
    public <T,R> Query createQuery(LambdaQuery<T,R> ... lambdaQueries){
        return createQuery(QueryUtils.getColumns(lambdaQueries));
    }
    @Override
    public <T,R> Query createQuery(Class<?> clazz, LambdaQuery<T,R> ... names){
        return createQuery(clazz, QueryUtils.getColumns(names));
    }

    @Override
    public <T,R> Query createQuery(Select<T,R> select){

        return createQuery(QueryUtils.getColumns(
                select != null?select.getLambdaQueries().toArray(new LambdaQuery[select.size()]):null
                ));
    }
    @Override
    public <T,R> Query createQuery(Class<?> clazz, Select<T,R> select){
        return createQuery(clazz, QueryUtils.getColumns(
                select != null?select.getLambdaQueries().toArray(new LambdaQuery[select.size()]):null
            ));
    }

    @Override
    public Query createQuery(Class<?> clazz, String... names){
        this.returnClass = clazz;
        this.returnColumns.clear();
        this.fixedReturnColumns.clear();
        if(names != null){
            for(String name : names){
                returnColumns.add(new Column(name));
            }
        }
        return this;
    }

    @Override
    public Query createQuery(Class<?> clazz, Column... columns){
        this.returnClass = clazz;
        this.returnColumns.clear();
        this.fixedReturnColumns.clear();
        if(columns != null){
            for(Column column : columns){
                returnColumns.add(column);
            }
        }
        return this;
    }

    @Override
    public Query addReturnColumn(Column ... columns){
        if(columns != null && columns.length > 0) {
            for (int i = 0; i < columns.length; i++) {
                this.returnColumns.add(columns[i]);
            }
        }
        return this;
    }

    @Override
    public Query addReturnColumn(String ... columns) {
        if(columns != null && columns.length > 0) {
            for (int i = 0; i < columns.length; i++) {
                this.returnColumns.add(new Column(columns[i]));
            }
        }
        return this;
    }

    @Override
    public Query addReturnColumn(Columns ... columns) {
        if (columns != null && columns.length > 0){
            for (int i = 0; i < columns.length; i++) {
                this.returnColumns.addAll(columns[i].getColumnList());
            }
        }
        return this;
    }

    @Override
    public Query clearReturnColumns(){
        this.returnColumns.clear();
        return this;
    }

    @Override
    public Query addFixedReturnColumn(Column ... columns){
        if(columns != null && columns.length > 0) {
            for (int i = 0; i < columns.length; i++) {
                this.fixedReturnColumns.add(columns[i]);
            }
        }
        return this;
    }

    @Override
    public <T,R> Query addReturnColumn(LambdaQuery<T,R> ... fieldFunctions){
        if(fieldFunctions != null && fieldFunctions.length > 0) {
            for (int i = 0; i < fieldFunctions.length; i++) {
                this.addReturnColumn(new Column(fieldFunctions[i]));
            }
        }
        return this;
    }
    @Override
    public <T,R> Query addFixedReturnColumn(LambdaQuery<T,R> ... fieldFunctions){
        if(fieldFunctions != null && fieldFunctions.length > 0) {
            for (int i = 0; i < fieldFunctions.length; i++) {
                this.addFixedReturnColumn(new Column(fieldFunctions[i]));
            }
        }
        return this;
    }

    @Override
    public <T,R> Query addReturnColumn(Select<T,R> select){
        if (select != null) {
            List<LambdaQuery<T, R>> lambdaQueries = select.getLambdaQueries();
            for (int i = 0; i < lambdaQueries.size(); i++) {
                this.addReturnColumn(new Column(lambdaQueries.get(i)));
            }
        }
        return this;
    }
    @Override
    public <T,R> Query addFixedReturnColumn(Select<T,R> select){
        if (select != null) {
            List<LambdaQuery<T, R>> lambdaQueries = select.getLambdaQueries();
            for (int i = 0; i < lambdaQueries.size(); i++) {
                this.addFixedReturnColumn(new Column(lambdaQueries.get(i)));
            }
        }
        return this;
    }

    @Override
    public Query addFixedReturnColumn(String ... columns){
        if(columns != null && columns.length > 0) {
            for (int i = 0; i < columns.length; i++) {
                this.fixedReturnColumns.add(new Column(columns[i]));
            }
        }
        return this;
    }

    @Override
    public Query addFixedReturnColumn(Columns ... columns) {
        if(columns != null && columns.length > 0) {
            for (int i = 0; i < columns.length; i++) {
                this.fixedReturnColumns.addAll(columns[i].getColumnList());
            }
        }
        return this;
    }

    @Override
    public Query clearFixedReturnColumns(){
        this.returnColumns.clear();
        return this;
    }


    @Override
    public Query createQuery(String... names){
        this.returnClass = this.table;
        this.returnColumns.clear();
        this.fixedReturnColumns.clear();
        if(names != null){
            for(String name : names){
                returnColumns.add(new Column(name));
            }
        }
        return this;
    }

    public Query createQuery(Column... columns) {
        this.returnClass = this.table;
        this.returnColumns.clear();
        this.fixedReturnColumns.clear();
        if(columns != null){
           for(Column column : columns){
               returnColumns.add(column);
           }
        }
        return this;
    }

    @Override
    public Query distinct() {
        this.distinct = true;
        return this;
    }


    public Query table(Class<?> clazz){
        if(clazz != null) {
            RdTable rdTable = SpringUtils.findAnnotation(clazz, RdTable.class);
            if(rdTable != null) {
                this.table = clazz;
                this.returnClass = clazz;
            }
        }
		return this;
	}


    @Override
    public Class<?> getTable() {
        return table;
    }

    @Override
    public List<Column> getGroups() {
        return groups;
    }

    @Override
    public Class<?> getReturnClass() {
        return returnClass;
    }

    @Override
    public List<Column> getReturnColumns() {
        List<Column> columnList = new ArrayList<Column>();
        if(returnColumns.isEmpty()){
            columnList.add(new Column(Column.ALL));
        }else{
            columnList.addAll(returnColumns);
        }
        return columnList;
    }

    @Override
    public List<Column> getFinalReturnColumns() {
        List<Column> columnList = new ArrayList<Column>();
        columnList.addAll(getReturnColumns());
        columnList.addAll(fixedReturnColumns);
        return columnList;
    }

    @Override
    public List<Column> getFixedReturnColumns() {
        return fixedReturnColumns;
    }

    public QueryInfo doQuery() {
        return options.doQuery(this, getPaging());
    }

    public QueryInfo doQueryCount() {
        return options.doQueryCount(this);
    }

}
