/*
 * Copyright 2017 @objectsql.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.objectsql.spring;

import com.objectsql.ObjectSQLManager;
import com.objectsql.support.CommitHandler;
import com.objectsql.support.ExecuteHandler;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import javax.sql.DataSource;

public class TxCommit {

    private PlatformTransactionManager transactionManager;
    private TransactionStatus status;
    private boolean executeSuccess = false;
    private CommitHandler commitHandler;
    private int propagation;
    private int isolationLevel;

    public TxCommit(PlatformTransactionManager manager){
        this(manager, TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(PlatformTransactionManager manager, int propagation){
        this(manager, propagation, -1);
    }

    public TxCommit(PlatformTransactionManager manager, int propagation, int isolationLevel){
        this.transactionManager = manager;
        this.propagation = propagation;
        this.isolationLevel = isolationLevel;
    }

    public TxCommit(ObjectSQLManager sqlManager){
        this(sqlManager.getDataSource(), TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(ObjectSQLManager sqlManager, int propagation){
        this(sqlManager.getDataSource(), propagation, -1);
    }

    public TxCommit(ObjectSQLManager sqlManager, int propagation, int isolationLevel){
        this(new DataSourceTransactionManager(sqlManager.getDataSource()), propagation, isolationLevel);
    }

    public TxCommit(DataSource dataSource){
        this(dataSource, TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(DataSource dataSource, int propagation){
        this(dataSource, propagation, -1);
    }

    public TxCommit(DataSource dataSource, int propagation, int isolationLevel){
        this(new DataSourceTransactionManager(dataSource), propagation, isolationLevel);
    }

    public <T> T execute(ExecuteHandler<T> handler){
        return execute(handler, null);
    }
    public <T> T execute(ExecuteHandler<T> handler, CommitHandler<T> commitHandler){
        T result = null;
        if(handler == null){
            return result;
        }
        try{
            manual();
            result = handler.execute();
            if (handler.shouldRollback(result)){
                rollback();
            }else {
                commit();
            }
        }catch (Exception e){
            if(commitHandler != null){
                commitHandler.handle(result, e);
            }
        }finally {
            if(!executeSuccess) {
                rollback();
            }
        }
        return result;
    }

    public boolean isExecuteSuccess(){
        return this.executeSuccess;
    }

    private void manual(){
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
        definition.setPropagationBehavior(this.propagation);
        definition.setIsolationLevel(this.isolationLevel);
        status = this.transactionManager.getTransaction(definition);
        executeSuccess = false;
    }

    private void commit(){
        if(transactionManager != null && status != null ){
            transactionManager.commit(status);
            executeSuccess = true;
        }
    }

    private void rollback(){
        if(!executeSuccess && transactionManager != null && status != null ){
            transactionManager.rollback(status);
        }
    }

    public static <T> T tx(DataSource dataSource, ExecuteHandler<T> handler) {
        return new TxCommit(dataSource).execute(handler);
    }

    public static <T> T tx(DataSource dataSource, int propagation, ExecuteHandler<T> handler) {
        return new TxCommit(dataSource, propagation).execute(handler);
    }

    public static <T> T tx(DataSource dataSource,int propagation, int isolationLevel, ExecuteHandler<T> handler) {
        return new TxCommit(dataSource, propagation, isolationLevel).execute(handler);
    }

    public static <T> T tx(DataSource dataSource, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSource).execute(handler, commitHandler);
    }

    public static <T> T tx(DataSource dataSource, int propagation, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSource, propagation).execute(handler, commitHandler);
    }

    public static <T> T tx(DataSource dataSource, int propagation, int isolationLevel, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSource, propagation, isolationLevel).execute(handler, commitHandler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager, ExecuteHandler<T> handler) {
        return new TxCommit(objectSQLManager).execute(handler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager, int propagation, ExecuteHandler<T> handler) {
        return new TxCommit(objectSQLManager, propagation).execute(handler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager,int propagation, int isolationLevel, ExecuteHandler<T> handler) {
        return new TxCommit(objectSQLManager, propagation, isolationLevel).execute(handler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(objectSQLManager).execute(handler, commitHandler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager, int propagation, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(objectSQLManager, propagation).execute(handler, commitHandler);
    }

    public static <T> T tx(ObjectSQLManager objectSQLManager, int propagation, int isolationLevel, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(objectSQLManager, propagation, isolationLevel).execute(handler, commitHandler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager, ExecuteHandler<T> handler) {
        return new TxCommit(dataSourceTransactionManager).execute(handler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, ExecuteHandler<T> handler) {
        return new TxCommit(dataSourceTransactionManager, propagation).execute(handler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager,int propagation, int isolationLevel, ExecuteHandler<T> handler) {
        return new TxCommit(dataSourceTransactionManager, propagation, isolationLevel).execute(handler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSourceTransactionManager).execute(handler, commitHandler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSourceTransactionManager, propagation).execute(handler, commitHandler);
    }

    public static <T> T tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, int isolationLevel, ExecuteHandler<T> handler, CommitHandler<T> commitHandler) {
        return new TxCommit(dataSourceTransactionManager, propagation, isolationLevel).execute(handler, commitHandler);
    }

}
