package com.paystack.android_sdk.ui.paymentchannels.card.cardform.views

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.core.api.models.CardParams
import com.paystack.android_sdk.core.logging.Logger
import com.paystack.android_sdk.ui.data.transaction.TransactionRepository
import com.paystack.android_sdk.ui.models.Charge
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class CardDetailsViewModel(
    private val transactionAccessData: AccessCodeData,
    private val transactionRepository: TransactionRepository,
) : ViewModel() {
    private val _state = MutableStateFlow(
        CardDetailsState(
            currency = transactionAccessData.currency,
            amount = transactionAccessData.amount,
            isTestTransaction = transactionAccessData.isTestTransaction,
            showTestCards = transactionAccessData.isTestTransaction,
        )
    )
    val state: StateFlow<CardDetailsState> = _state.asStateFlow()

    private val _result = MutableStateFlow<Result<Charge>?>(null)
    val result = _result.asStateFlow()

    fun setCardNumber(cardNumber: String) {
        _state.update {
            val cardData = it.cardData.copy(cardNumber = cardNumber)
            it.copy(cardData = cardData)
        }
    }

    fun setCardExpiry(cardExpiry: String) {
        _state.update {
            val cardData = it.cardData.copy(cardExpiry = cardExpiry)
            it.copy(cardData = cardData)
        }
    }

    fun setCardCVV(cvv: String) {
        _state.update {
            val cardData = it.cardData.copy(cardCVV = cvv)
            it.copy(cardData = cardData)
        }
    }

    fun submitCardDetails() {
        viewModelScope.launch {
            _state.update { it.copy(isSubmitting = true) }
            val currentState = _state.value
            val cardParams = if (currentState.showTestCards) {
                currentState.selectedTestCard?.toCardParams() ?: run {
                    //TODO: Figure out what to do about errors that should be fatal
                    Logger.error("No test card selected", IllegalStateException())
                    null
                }
            } else {
                val (expMonth, expYear) = expiryDateComponents(state.value.cardData.cardExpiry)
                CardParams(
                    number = state.value.cardData.cardNumber,
                    cvv = state.value.cardData.cardCVV,
                    expiryMonth = expMonth,
                    expiryYear = expYear,
                )
            }

            if (cardParams != null) {
                _result.value = transactionRepository.chargeCard(
                    accessCode = transactionAccessData.accessCode,
                    cardParams = cardParams,
                    encryptionKey = transactionAccessData.encryptionKey,
                )
            }
            _state.update { it.copy(isSubmitting = false) }
        }
    }

    fun setSelectedTestCard(index: Int) {
        _state.update { it.copy(selectedTestCardIndex = index) }
    }

    fun setShowTestCards(showTestCards: Boolean) {
        _state.update { it.copy(showTestCards = showTestCards) }
    }

    fun toggleShowTestCards() {
        setShowTestCards(!state.value.showTestCards)
    }

    private fun expiryDateComponents(expiryDate: String): Pair<String, String> {
        val month = expiryDate.substring(0, 2)
        val year = expiryDate.substring(expiryDate.length - 2, expiryDate.length)
        return Pair(month, year)
    }
}
