package com.sappenin.utils;

import java.nio.ByteBuffer;
import java.util.UUID;

import org.apache.commons.codec.binary.Base64;

/**
 * A utility class that generates globally unique strings by base-64-endcoding a
 * {@link UUID} into a 22 character long, web-safe {@link String}. This class's
 * output may safely be used in any URL without encoding. The output also
 * provides a shorter (yet equivalent) value as its corresponding UUID while
 * maintaining all of the characteristics of a UUID.
 */
public class UrlSafe22
{

	/**
	 * Convert an instance of {@link UUID} into a 22 character long, base-64
	 * encoded {@link String} equivalent.
	 * 
	 * @param uuid An instance of {@link UUID} to convert from.
	 * @return
	 */
	public static String fromUUID(final UUID uuid)
	{
		final Base64 base64 = new Base64();
		final ByteBuffer byteBuffer = ByteBuffer.wrap(new byte[16]);

		byteBuffer.putLong(uuid.getMostSignificantBits());
		byteBuffer.putLong(uuid.getLeastSignificantBits());

		return base64.encodeBase64URLSafeString(byteBuffer.array());
	}

	/**
	 * Decodes a base64-encoded {@link String} into its equivalent {@link UUID}.
	 * 
	 * @param urlSafe22 A 22 character long base64-encoded {@link String}
	 *            representing an encoded UUID.
	 * @return
	 */
	public static UUID toUUID(final String urlSafe22)
	{
		final Base64 base64 = new Base64();
		final byte[] bytes = base64.decodeBase64(urlSafe22);
		final ByteBuffer bb = ByteBuffer.wrap(bytes);
		return new UUID(bb.getLong(), bb.getLong());
	}

}
