/**
 * Copyright (C) 2014 Sappenin Inc. (developers@sappenin.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.sappenin.utils.json;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.logging.Logger;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.io.CharStreams;

/**
 * Utilities for Dealing with JSON (via Jackson).
 *
 * @author David Fuelling
 */
public interface JsonUtils
{
	/**
	 * Convert an object of type
	 * <P>
	 * into a Json String.
	 *
	 * @param payload
	 * @return
	 * @throws JsonProcessingException
	 */
	<P> String toJson(P payload) throws JsonProcessingException;

	/**
	 * Convert an object of type
	 * <P>
	 * into a Json String. Catch any exceptions of type
	 * {@link JsonProcessingException} and throw them as
	 * {@link RuntimeException}.
	 *
	 * @param payload
	 * @return
	 * @throws RuntimeException
	 */
	<P> String toJsonUnchecked(P payload) throws RuntimeException;

	/**
	 * Convert the body of an {@link HttpServletRequest} into an of the proper
	 * type as based upon an instance of {@link JsonUtilsClassTypeMapper}.
	 *
	 * @param httpServletRequest An {@link HttpServletRequest} with an
	 *            {@link InputStream} containing a JSON payload.
	 * @param jsonUtilsClassTypeMapper An instance of
	 *            {@link JsonUtilsClassTypeMapper} to map the incoming JSON
	 *            payload to a typed Java class.
	 * @return
	 * @throws IOException
	 * @throws ClassNotFoundException
	 */
	<P> P fromJson(final HttpServletRequest httpServletRequest, final JsonUtilsClassTypeMapper jsonUtilsClassTypeMapper)
			throws IOException, ClassNotFoundException;

	/**
	 * Convert a {@link JsonNode} into an object of the proper type as found in
	 * the "@class" directive in the JsonNode.
	 *
	 * @param jsonNode A {@link JsonNode} to deserialize into a Java object.
	 * @param jsonNodeClass The expected Java {@link Class} to deserialize the
	 *            incoming JSON payload to.
	 * @return
	 * @throws ClassNotFoundException
	 */
	<P> P fromJson(JsonNode jsonNode, final Class<P> jsonNodeClass) throws ClassNotFoundException;

	/**
	 * Convert the body of an {@link HttpServletRequest} into an of the proper
	 * type as based upon an instance of {@link JsonUtilsClassTypeMapper}.
	 *
	 * @param jsonNode A {@link JsonNode} to deserialize into a Java object.
	 * @param jsonUtilsClassTypeMapper An instance of
	 *            {@link JsonUtilsClassTypeMapper} to map the incoming JSON
	 *            payload to a typed Java class.
	 * @param <P>
	 * @return
	 * @throws ClassNotFoundException
	 */
	<P> P fromJson(final JsonNode jsonNode, final JsonUtilsClassTypeMapper jsonUtilsClassTypeMapper)
			throws ClassNotFoundException;

	/**
	 * An implementation of {@link JsonUtils}.
	 *
	 * @author dfuelling
	 */
	public static class Impl implements JsonUtils
	{
		protected final static Logger logger = Logger.getLogger(Impl.class.getName());

		private final ObjectMapper objectMapper;

		/**
		 * Required-Args constructor
		 *
		 * @param objectMapper
		 */
		public Impl(final ObjectMapper objectMapper)
		{
			this.objectMapper = objectMapper;
		}

		/**
		 *
		 * @param payload
		 * @return
		 * @throws JsonProcessingException
		 */
		@Override
		public <T> String toJson(final T payload) throws JsonProcessingException
		{
			Preconditions.checkNotNull(payload);

			final String jsonString = this.objectMapper.writeValueAsString(payload);
			return jsonString;
		}

		@Override
		public <P> String toJsonUnchecked(P payload) throws RuntimeException
		{
			try
			{
				return this.toJson(payload);
			}
			catch (JsonProcessingException jpe)
			{
				throw new RuntimeException(jpe);
			}
		}

		// //////////////////////////
		// From JSON
		// //////////////////////////

		@Override
		public <P> P fromJson(final JsonNode jsonNode, final JsonUtilsClassTypeMapper jsonUtilsClassTypeMapper)
				throws ClassNotFoundException
		{
			Preconditions.checkNotNull(jsonNode);
			Preconditions.checkNotNull(jsonUtilsClassTypeMapper);

			Class<P> jsonPayloadClass = jsonUtilsClassTypeMapper.getJsonClassType(jsonNode);
			return this.fromJson(jsonNode, jsonPayloadClass);
		}

		@Override
		public <P> P fromJson(final JsonNode jsonNode, final Class<P> jsonNodeClass) throws ClassNotFoundException
		{
			Preconditions.checkNotNull(jsonNode);
			Preconditions.checkNotNull(jsonNodeClass);

			// The Payload is UTF-8.
			logger.info(String.format("Converting the JsonNode \"%s\" to a Java Class of type: \"%s\"))", jsonNode,
				jsonNodeClass.getName()));

			final P payload = this.objectMapper.convertValue(jsonNode, jsonNodeClass);
			return payload;
		}

		@Override
		public <P> P fromJson(final HttpServletRequest httpServletRequest,
				final JsonUtilsClassTypeMapper jsonUtilsClassTypeMapper) throws IOException, ClassNotFoundException
		{
			Preconditions.checkNotNull(httpServletRequest);

			JsonNode jsonNode = this.getRequestPayloadAsJsonNode(httpServletRequest);

			return this.fromJson(jsonNode, jsonUtilsClassTypeMapper);
		}

		// ////////////////////////////
		// Private Helpers
		// ////////////////////////////

		/**
		 * Retrieves the {@link JsonNode} from the {@code request}.
		 *
		 * @param request
		 * @return
		 * @throws IOException
		 * @throws ClassNotFoundException
		 */
		@VisibleForTesting
		protected JsonNode getRequestPayloadAsJsonNode(final HttpServletRequest request) throws IOException,
				ClassNotFoundException
		{
			Preconditions.checkNotNull(request);

			// The Payload must be UTF-8!
			String jsonPayload = this.getRequestPayloadAsJsonString(request);

			// since 2.1 use mapper.getFactory() instead
			final JsonParser jp = this.objectMapper.getFactory().createParser(jsonPayload);
			final JsonNode jsonNode = this.objectMapper.readTree(jp);
			return jsonNode;
		}

		/**
		 * Helper method to grab a Json Payload from an
		 * {@link HttpServletRequest} by forwarding to
		 * {@link #getRequestPayloadAsJsonString(InputStream)}.
		 *
		 * @param httpServletRequest An instance of {@link HttpServletRequest}.
		 *
		 * @return
		 */
		@VisibleForTesting
		protected String getRequestPayloadAsJsonString(final HttpServletRequest httpServletRequest) throws IOException
		{
			Preconditions.checkNotNull(httpServletRequest);
			return this.getRequestPayloadAsJsonString(httpServletRequest.getInputStream());
		}

		/**
		 * Helper method to grab a Json Payload from an {@link InputStream}.
		 * This is generally used in concert with an {@link HttpServletRequest},
		 * but doesn't strictly need to be. Not used in this class but used by
		 * sub-classes.
		 *
		 * @param inputStream An instance of {@link InputStream}.
		 *
		 * @return
		 */
		@VisibleForTesting
		protected String getRequestPayloadAsJsonString(final InputStream inputStream) throws IOException
		{
			Preconditions.checkNotNull(inputStream);

			// The Payload must be UTF-8!
			String jsonPayload = "";
			try (final InputStream stream = inputStream)
			{
				jsonPayload = CharStreams.toString(new InputStreamReader(stream, Charsets.UTF_8));
				return jsonPayload;
			}
			finally
			{
				logger.info(String.format("Retrieved the following JSON Payload from HttpServletRequest: %s", jsonPayload));
				//logger.exiting(this.getClass().getName(), "getJsonPayloadFromRequest", jsonPayload);
			}
		}
	}
}
