package com.sappenin.utils.json;

import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Preconditions;

/**
 * An implementation of {@link JsonUtilsClassTypeMapper} that inspects a "kind"
 * attribute in a {@link JsonTypeInfo} annotation that exists on a particular
 * JSON serialized class.
 */
public class JsonUtilsClassTypeMapperForJackson implements JsonUtilsClassTypeMapper
{
	public static final String KIND = "kind";
	public static final String AT_CLASS = "@class";

	private final String jsonTypeAttributeName;

	/**
	 * No-Args Constructor.
	 */
	public JsonUtilsClassTypeMapperForJackson()
	{
		this.jsonTypeAttributeName = AT_CLASS;
	}

	/**
	 * Required-Args Constructor.
	 *
	 * @param jsonTypeAttributeName A {@link String} representing the name of
	 *            the {@link JsonTypeInfo} value for the attribute that will
	 *            identify the class type of a particular serializaed Json
	 *            object.
	 */
	public JsonUtilsClassTypeMapperForJackson(final String jsonTypeAttributeName)
	{
		Preconditions.checkNotNull(jsonTypeAttributeName);
		this.jsonTypeAttributeName = jsonTypeAttributeName;
	}

	@Override
	public Class<?> getJsonClassType(final JsonNode jsonNode)
	{
		JsonNode classNameJsonNode = jsonNode.get(jsonTypeAttributeName);
		Preconditions.checkNotNull(classNameJsonNode, String.format(
			"Cannot DeSerialize polymorphic JSON payload via Jackson when the Payload has no "
				+ "@JsonTypeInfo with 'property' value of '%s' specified!", jsonTypeAttributeName));
		try
		{
			Class<?> payloadClass = Class.forName(classNameJsonNode.asText());
			return payloadClass;
		}
		catch (ClassNotFoundException e)
		{
			throw new RuntimeException(e);
		}
	}
}
