/**
 * Copyright (C) 2015 Sappenin Inc. (developers@sappenin.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.sappenin.utils.servletfilters;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Set;
import java.util.logging.Logger;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

/**
 * A Servlet Filter that logs incoming headers for the request.
 */
public abstract class RequestHeadersLoggingFilter implements Filter
{
	/**
	 * A {@link Set} of obfuscated header names that this filter will obfuscate
	 * before logging.
	 * 
	 * @return
	 */
	protected abstract Set<String> getObfuscatedHeaderNames();

	/**
	 * Accessor for implementations to provide their own logger implementation.
	 * 
	 * @return
	 */
	protected abstract Logger getLogger();

	/**
	 * Log all Mashape Headers...
	 * 
	 * @param servletRequest
	 * @param servletResponse
	 * @param filterChain
	 */
	public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain)
			throws IOException, ServletException
	{
		if (servletRequest instanceof HttpServletRequest)
		{
			final HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;

			StringBuilder sbHeaders = new StringBuilder();
			sbHeaders.append("Request Headers: ");

			final Enumeration headerNames = httpServletRequest.getHeaderNames();
			while (headerNames.hasMoreElements())
			{
				final String headerValue;
				final String headerName = headerNames.nextElement().toString();
				if (headerName != null)
				{
					// Obfuscate any sensitive headers
					if (getObfuscatedHeaderNames().contains(headerName.toLowerCase(Locale.ENGLISH)))
					{
						headerValue = "**********";
					}
					else
					{
						headerValue = httpServletRequest.getHeader(headerName);
					}
					sbHeaders.append(String.format("\n%s: %s", headerName, headerValue));
				}
			}

			getLogger().info(sbHeaders.toString());
		}

		filterChain.doFilter(servletRequest, servletResponse);
	}
}
