/*
 * Decompiled with CFR 0.152.
 */
package com.schematic.api;

import com.schematic.api.logger.SchematicLogger;
import com.schematic.api.resources.events.EventsClient;
import com.schematic.api.resources.events.requests.CreateEventBatchRequestBody;
import com.schematic.api.types.CreateEventRequestBody;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class EventBuffer
implements AutoCloseable {
    private static final Duration DEFAULT_FLUSH_INTERVAL = Duration.ofMillis(5000L);
    private static final int DEFAULT_MAX_BATCH_SIZE = 100;
    private static final int DEFAULT_MAX_QUEUE_SIZE = 10000;
    private static final int MAX_RETRY_ATTEMPTS = 3;
    private static final Duration RETRY_INITIAL_DELAY = Duration.ofMillis(100L);
    private final ConcurrentLinkedQueue<CreateEventRequestBody> events = new ConcurrentLinkedQueue();
    private final int maxBatchSize;
    private final Duration flushInterval;
    private final EventsClient eventsClient;
    private final SchematicLogger logger;
    private final ScheduledExecutorService scheduler;
    private final AtomicInteger droppedEvents;
    private final AtomicInteger processedEvents;
    private final AtomicInteger failedEvents;
    private volatile boolean stopped;

    public EventBuffer(EventsClient eventsClient, SchematicLogger logger, int maxBatchSize, Duration flushInterval) {
        this.maxBatchSize = maxBatchSize > 0 ? maxBatchSize : 100;
        this.flushInterval = flushInterval != null ? flushInterval : DEFAULT_FLUSH_INTERVAL;
        this.eventsClient = eventsClient;
        this.logger = logger;
        this.droppedEvents = new AtomicInteger(0);
        this.processedEvents = new AtomicInteger(0);
        this.failedEvents = new AtomicInteger(0);
        this.stopped = false;
        this.scheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                Thread thread = new Thread(r, "SchematicEventBuffer");
                thread.setDaemon(true);
                return thread;
            }
        });
        this.startPeriodicFlush();
    }

    public void push(CreateEventRequestBody event) {
        if (event == null) {
            this.logger.warn("Attempted to push null event to buffer", new Object[0]);
            return;
        }
        if (this.stopped) {
            this.logger.error("Event buffer is stopped, not accepting new events", new Object[0]);
            return;
        }
        if (this.events.size() >= 10000) {
            this.droppedEvents.incrementAndGet();
            this.logger.warn("Event buffer queue size exceeded, dropping oldest event", new Object[0]);
            this.events.poll();
        }
        this.events.offer(event);
        if (this.events.size() >= this.maxBatchSize) {
            this.flush();
        }
    }

    public void flush() {
        if (this.events.isEmpty()) {
            return;
        }
        ArrayList<CreateEventRequestBody> batch = new ArrayList<CreateEventRequestBody>();
        while (!this.events.isEmpty() && batch.size() < this.maxBatchSize) {
            CreateEventRequestBody event = this.events.poll();
            if (event == null) continue;
            batch.add(event);
        }
        if (!batch.isEmpty()) {
            this.sendBatchWithRetry(batch, 0);
        }
    }

    private void sendBatchWithRetry(List<CreateEventRequestBody> batch, int retryCount) {
        try {
            CreateEventBatchRequestBody requestBody = CreateEventBatchRequestBody.builder().events(batch).build();
            this.eventsClient.createEventBatch(requestBody);
            this.processedEvents.addAndGet(batch.size());
        }
        catch (Exception e) {
            if (retryCount < 3) {
                long delayMillis = RETRY_INITIAL_DELAY.toMillis() * (1L << retryCount);
                this.logger.warn("Failed to send event batch, attempting retry %d of %d in %d ms", retryCount + 1, 3, delayMillis);
                this.scheduler.schedule(() -> this.sendBatchWithRetry(batch, retryCount + 1), delayMillis, TimeUnit.MILLISECONDS);
            }
            this.failedEvents.addAndGet(batch.size());
            this.logger.error("Failed to flush events: " + e.getMessage(), new Object[0]);
        }
    }

    private void startPeriodicFlush() {
        this.scheduler.scheduleAtFixedRate(() -> {
            try {
                this.flush();
            }
            catch (Exception e) {
                this.logger.error("Error during periodic flush: %s", e.getMessage());
            }
        }, this.flushInterval.toMillis(), this.flushInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() {
        if (!this.stopped) {
            this.stopped = true;
            try {
                this.flush();
                this.scheduler.shutdown();
                if (!this.scheduler.awaitTermination(30L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                    if (!this.scheduler.awaitTermination(30L, TimeUnit.SECONDS)) {
                        this.logger.error("EventBuffer thread pool did not terminate", new Object[0]);
                    }
                }
            }
            catch (InterruptedException e) {
                this.scheduler.shutdownNow();
                Thread.currentThread().interrupt();
                this.logger.error("EventBuffer shutdown interrupted", new Object[0]);
            }
        }
    }

    public String getMetrics() {
        return String.format("EventBuffer Metrics - Processed: %d, Dropped: %d, Failed: %d, Current Queue Size: %d", this.processedEvents.get(), this.droppedEvents.get(), this.failedEvents.get(), this.events.size());
    }

    public int getQueueSize() {
        return this.events.size();
    }

    public int getProcessedCount() {
        return this.processedEvents.get();
    }

    public int getDroppedCount() {
        return this.droppedEvents.get();
    }

    public int getFailedCount() {
        return this.failedEvents.get();
    }
}

