/*
 * This file is distributed as part of the MariaDB Manager. It is free
 * software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Copyright 2012-2014 SkySQL Corporation Ab
 * 
 * Author: Mark Riddoch, Massimo Siani
 * Date: February 2013
 */

package com.skysql.java;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.lang.reflect.Method;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.skysql.java.Configuration.DEFAULT_SECTION;

/**
 * The interface to the SkySQL Manager API, a REST interface to the monitor database.
 * This is the API that should be used to store all monitor observations and to retrieve 
 * system configuration information rather than access the SQLite monitoring database directly.
 * 
 * @author Mark Riddoch
 * @author Massimo Siani
 *
 */
public class MonAPI {
	/**
	 * The host on which the API is running
	 */
	String		m_apiHost;
	/**
	 * The API Key to use to access the API
	 */
	String		m_apiKey;
	/**
	 * The API key ID as assigned by the API provider
	 */
	String		m_apiKeyID;
	/**
	 * Execute stored batch commands?
	 */
	private boolean		m_bufferingExecution;
	/**
	 * Count the number of calls, to avoid buffering too often
	 */
	private int			m_cycles;
	/**
	 * Timezone read from the php configuration. Assume /etc/php.ini, default to Europe/London.
	 */
	private String		m_timeZone = null;

	/**
	 * Constructs the MonAPI instance. This consists of obtaining the information required
	 * to contact the API. This information is available via Java System Properties.
	 * Sets the buffering to true by default, and the key ID to 1.
	 */
	public MonAPI()
	{
		this(true, 1);
	}
	/**
	 * Constructs the MonAPI instance. This consists of obtaining the information required
	 * to contact the API. This information is available via Java System Properties.
	 * Sets the buffering to true by default.
	 * 
	 * @param APIKeyID		the API key ID of the running component
	 */
	public MonAPI(int APIKeyID) {
		this(true, APIKeyID);
	}
	/**
	 * Constructs the MonAPI instance. This consists of obtaining the information required
	 * to contact the API. This information is available via Java System Properties.
	 *
	 * @param buffer		true if buffer is active (default).
	 * @param APIKeyId		the API key ID of the running component. Default is 1.
	 */
	public MonAPI(boolean buffer, int APIKeyId) {
		getApiCredentials(APIKeyId);
		m_bufferingExecution = buffer;
		m_cycles = 0;
	}

	/**
	 * Retrieves the API host and ID/key pair according to the following logic.
	 * If the SKYSQL_API_HOST variable is set, it is used as the host.
	 * If no such variable is specified, tries to look in the configuration file
	 * /usr/local/skysql/config/manager.json,
	 * in JSON format, an object of type GsonConfig which contains the api.uri object.
	 * If it is still not found, uses localhost.
	 * Similar considerations hold for SKYSQL_API_KEYID (API ID) and SKYSQL_API_KEY (API key),
	 * but the configuration file looked for is
	 * /usr/local/skysql/config/components.ini.
	 * The resulting data is stored as the state of the current instance.
	 * 
	 * @param APIKeyId				the API ID to be used if none is provided as system property
	 */
	private void getApiCredentials(int APIKeyId) {
		Properties props = System.getProperties();
		m_apiHost = props.getProperty("SKYSQL_API_HOST");
		m_apiKey = props.getProperty("SKYSQL_API_KEY");
		m_apiKeyID = props.getProperty("SKYSQL_API_KEYID");
		if (m_apiKeyID != null && ! m_apiKeyID.isEmpty()) {
			try {
				APIKeyId = Integer.parseInt(m_apiKeyID);
			} catch (Exception e) {
				Logging.error("The provided API Key ID is invalid. Looking for another value in the configuration files.");
				m_apiKeyID = null;
			}
		}
		if (isPropertyNullOrEmpty()) {
			getConfig();
		}
		if (isPropertyNullOrEmpty()) {
			getConfigJson(APIKeyId);
		}
		if (isPropertyNullOrEmpty()) {
			getConfigIni(APIKeyId);
		}
		if (isPropertyNullOrEmpty()) {
			getConfigDefault(APIKeyId);
		}
	}
	/**
	 * Checks whether any of the system properties in this instance is null or empty.
	 * @return		true if any property is null or empty, false otherwise
	 */
	private boolean isPropertyNullOrEmpty() {
		boolean result = (m_apiHost == null || m_apiHost.isEmpty()
				|| m_apiKey == null || m_apiKey.isEmpty()
				|| m_apiKeyID == null || m_apiKeyID.isEmpty());
		return result;
	}
	/**
	 * Retrieves the API ID/key pair from the <code>Configuration</code> class.
	 */
	private void getConfig() {
		Configuration config = new Configuration();
		m_apiKeyID = config.getConfig(Configuration.getApplication()).get("apikeyid");
		m_apiKey = config.getConfig(DEFAULT_SECTION.APIKEYS).get(m_apiKeyID);
		m_apiHost = config.getConfig(DEFAULT_SECTION.APIHOST).get("uri");
	}
	/**
	 * If any of the system properties is null or empty, look for the Json configuration file.
	 * 
	 * @param APIKeyId				the API ID to be used
	 */
	private void getConfigJson(int APIKeyId) {
		String configJson = "";
		try {
			BufferedReader reader = new BufferedReader(new FileReader("/usr/local/skysql/config/manager.json"));
			String tmp;
			while ( (tmp = reader.readLine()) != null ) {
				configJson += tmp;
			}
			reader.close();
			GsonConfig gsonConfig = GsonManager.fromJson(configJson, GsonConfig.class);
			if (m_apiHost == null || m_apiHost.isEmpty()) {
				m_apiHost = gsonConfig.getUri();
			}
			if (m_apiKey == null || m_apiKey.isEmpty()) {
				m_apiKey = gsonConfig.getKey(APIKeyId);
			}
			if (m_apiKeyID == null || m_apiKeyID.isEmpty()) {
				m_apiKeyID = Integer.toString(APIKeyId);
			}
		} catch (Exception e) {
			Logging.info("Configuration Json file not found.");
		}
	}
	/**
	 * If any of the system properties is null or empty, look for the ini configuration file.
	 * 
	 * @param APIKeyId				the API ID to be used
	 */
	private void getConfigIni(int APIKeyId) {
		try {
			BufferedReader config = new BufferedReader(new FileReader("/usr/local/skysql/config/components.ini"));
			String keyString;
			while ((keyString = config.readLine()) != null) {
				Pattern p = Pattern.compile("^" + APIKeyId + "\\s*=\\s*\"(\\w+)\"");
				Matcher m = p.matcher(keyString);
				if (m.find()) {
					keyString = m.group(1);
					break;
				}
			}
			config.close();
			if (m_apiKey == null || m_apiKey.isEmpty()) {
				m_apiKey = keyString;
			}
			if (m_apiKeyID == null || m_apiKeyID.isEmpty()) {
				m_apiKeyID = "" + APIKeyId;
			}
		} catch (Exception e) {
			Logging.info("Configuration ini file not found.");
		}
	}
	/**
	 * Sets the default value for the unset properties. The API ID does not have a default
	 * value in the current implementation.
	 */
	private void getConfigDefault(int APIKeyId) {
		if (m_apiHost == null || m_apiHost.isEmpty()) {
			m_apiHost = "http://localhost/restfulapi";
		}
		if (m_apiKey == null || m_apiKey.isEmpty()) {
			m_apiKey = "1f8d9e040e65d7b105538b1ed0231770";
		}
		if (m_apiKeyID == null || m_apiKeyID.isEmpty()) {
			m_apiKeyID = "" + APIKeyId;
		}
	}
	/**
	 * Populate a monitor value for the system. Sends a POST request.
	 * 
	 * @param systemID		The ID of the System
	 * @param monitorKey	The key of the monitor itself
	 * @param value			The observed value
	 * @return True if the update was performed to the API
	 */
	public boolean MonitorValue(int systemID, String monitorKey, String value) {
		String apiRequest = "system/" + systemID + "/monitor/" + monitorKey + "/data";
		return restPost(apiRequest, new String[] {"value"}, new String[] {value});
	}
	/**
	 * Bulk update API. POST request.
	 * 
	 * @param apiRequest	the request uri
	 * @param fields
	 * @param values
	 * @return True if the update was performed
	 */
	public boolean bulkMonitorValue(String apiRequest, String[] fields, String[] values) {
		return restPost(apiRequest, fields, values);
	}

	/**
	 * API call which requires to modify something. Sends a PUT request.
	 * 
	 * @param restRequest	the API URI
	 * @param pName			the name of the parameter
	 * @param pValue		the value of the parameter
	 * @return				true on success, false otherwise
	 */
	public boolean updateValue(String restRequest, String pName, String pValue) {
		String[] newpName = {pName};
		String[] newpValue = {pValue};
		String result = updateValue(restRequest, newpName, newpValue);
		if (result == null) return false;
		return true;
	}

	/**
	 * API call which requires to modify something. Sends a PUT request.
	 * 
	 * @param restRequest	the API URI
	 * @param pName			an array with the names of the parameters
	 * @param pValue		an array with the values of the parameters
	 * @return				the Json from the API as is, or null if an error occurred.
	 */
	public String updateValue(String restRequest, String[] pName, String[] pValue) {
		String outJson = restPut(restRequest, pName, pValue);
		return outJson;
	}

	/**
	 * Bounce the Json that comes from the API. Only for GET requests.
	 * This is a convenient call for getReturnedJson(restRequest, pName, pValue, null).
	 * 
	 * @param restRequest		the API URI
	 * @param pName				an array with the names of the parameters, can be null
	 * @param pValue			an array with the values of the parameters, can be null
	 * @return					the Json from the API as is, or null if an error occurred.
	 */
	public String getReturnedJson(String restRequest, String[] pName, String[] pValue) {
		String outJson = getReturnedJson(restRequest, pName, pValue, null);
		return outJson;
	}

	/**
	 * Returns the Json that comes from the API. Only for GET requests.
	 * Sets the If-Modified-Since header, although it does not bounce
	 * the http code 304.
	 * 
	 * @param restRequest		the API URI
	 * @param pName				an array with the names of the parameters, can be null
	 * @param pValue			an array with the values of the parameters, can be null
	 * @param lastUpdate		the date for the If-Modified-Since header in RFC 2822 format, may be null or empty
	 * @return					the output Json, empty string if code 304 is returned
	 */
	public String getReturnedJson(String restRequest, String[] pName, String[] pValue, String lastUpdate) {
		String outJson = restGet(restRequest, pName, pValue, lastUpdate);
		if (outJson == null) {
			Logging.error("Failed: Output Json: " + outJson);
			Logging.debug("        URI request: " + restRequest);
			return null;
		}
		return outJson;
	}

	/**
	 * Send a GET request to the API and set the If-Modified-Since header.
	 * The lists of names and values of parameters to be passed to the API
	 * may and has to be null if no parameter is to be passed to the API.
	 * 
	 * @param restRequest	The URL, excluding the fixed stem
	 * @param pName[]		The parameter names for the GET request
	 * @param pValue[]		The parameter values for the GET request
	 * @param lastUpdate	The If-Modified-Since date, in RFC 2822 format
	 * @return				The output of the API (a JSON string)
	 */
	private String restGet(String restRequest, String[] pName, String[] pValue, String lastUpdate) {
		String result = "";
		String value = "";
		if (pName != null && pValue != null && ! pName[0].isEmpty()) {
			for (int i=0; i < pName.length; i++) {
				value += "&" + pName[i] + "=" + pValue[i];
			}
			value = value.substring(1);
		}
		try {
			// set up authorization
			String reqString = "" + m_apiHost + "/" + restRequest;
			if (value != "") {
				reqString += "?" + value;
			}
			String rfcdate = setDate();
			String sb = this.setAuth(restRequest, rfcdate);

			// set up connection
			URL postURL = new URL(reqString);
			HttpURLConnection apiConn = (HttpURLConnection) postURL.openConnection();
			if (lastUpdate != null && lastUpdate != "") {
				apiConn.setRequestProperty("If-Modified-Since", lastUpdate);
			}
			setUpConn(apiConn, sb, rfcdate, value, "GET");

			// get output
			BufferedReader in = new BufferedReader(new InputStreamReader(apiConn.getInputStream()));
			String tmp;
			while ((tmp = in.readLine()) != null)
				result += tmp + "\n";
			in.close();

			if (apiConn.getResponseCode() == 304) {
				result = "";
			}
		} catch (ConnectException e) {
			Logging.error("Cannot connect to the web server.");
			return "";
		} catch (Exception e) {
			Logging.error(e.getMessage());
			return "";
		}
		return result;
	}

	/**
	 * Send a PUT request to the API.
	 * 
	 * @param restRequest	The URL, excluding the fixed stem
	 * @param pName[]		The parameter names for the GET request
	 * @param pValue[]		The parameter values for the GET request
	 * @return				The output of the API (a JSON string)
	 */
	private String restPut(String restRequest, String[] pName, String[] pValue) {
		String result = "";
		String value = "";
		if (pName != null && pValue != null && ! pName[0].isEmpty()) {
			for (int i=0; i < pName.length; i++) {
				value += "&" + pName[i] + "=" + pValue[i];
			}
			value = value.substring(1);
		}
		try {
			// set up authorization
			String reqString = "" + m_apiHost + "/" + restRequest;
			String rfcdate = setDate();
			String sb = this.setAuth(restRequest, rfcdate);
			// set up connection
			URL postURL = new URL(reqString);
			HttpURLConnection apiConn = (HttpURLConnection) postURL.openConnection();
			setUpConn(apiConn, sb, rfcdate, value, "PUT");
			// get output
			BufferedReader in = new BufferedReader(new InputStreamReader(apiConn.getInputStream()));
			String tmp;
			while ((tmp = in.readLine()) != null) {
				result += tmp + "\n";
			}
			in.close();
			// run buffer
			runBuffer();
			if (apiConn.getResponseCode() != 200) {
				throw new RuntimeException("Failed : HTTP error : "
						+ apiConn.getResponseMessage() + ": returned data: " + result);
			}
		} catch (Exception e) {
			pushFailedApi("restPut", restRequest, pName, pValue);
			return null;
		}
		return result;
	}

	/**
	 * Send a POST request to the API.
	 * 
	 * @param restRequest	The URL, excluding the fixed stem
	 * @param pName			The parameter names for the post request
	 * @param pValue		The parameter values for the port request
	 */
	private boolean restPost(String restRequest, String[] pName, String[] pValue) {
		String result = "";
		String value = "";
		if (pName != null && pValue != null && ! pName[0].isEmpty()) {
			for (int i=0; i < pName.length; i++) {
				value += "&" + pName[i] + "=" + pValue[i];
			}
			value = value.substring(1);
		}
		try {
			// set up authorization for the redirected webpage (ie, $_POST variable)
			String reqString = "" + m_apiHost + "/" + restRequest;
			String rfcdate = setDate();
			String sb = this.setAuth(restRequest, rfcdate);
			// set up connection
			URL postURL = new URL(reqString);
			HttpURLConnection apiConn = (HttpURLConnection) postURL.openConnection();
			setUpConn(apiConn, sb, rfcdate, value, "POST");
			// get output
			BufferedReader in = new BufferedReader(new InputStreamReader(apiConn.getInputStream()));
			String tmp;
			while ((tmp = in.readLine()) != null) {
				result += tmp + "\n";
			}
			in.close();
			// buffer
			runBuffer();
			if (apiConn.getResponseCode() != HttpURLConnection.HTTP_OK) {
				throw new RuntimeException("Failed : HTTP error : "
						+ apiConn.getResponseMessage() + ": returned data: " + result);
			}
		} catch (Exception e) {
			pushFailedApi("restPost", restRequest, pName, pValue);
			return false;
		}
		return true;
	}










	/**
	 * Read the timezone configuration and set the appropriate date format.
	 * 
	 * @return a string with the appropriate date
	 * @throws IOException
	 */
	private String setDate() throws IOException {
		SimpleDateFormat sdf = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z");
		if (this.m_timeZone == null) {
			try {
				BufferedReader timeZone = new BufferedReader(new FileReader("/etc/php.ini"));
				String tz;
				while ((tz = timeZone.readLine()) != null) {
					Pattern p = Pattern.compile("date\\.timezone\\s*=\\s*(\\w+/\\w+)");
					Matcher m = p.matcher(tz);
					if (m.find()) {
						tz = m.group(1);
						break;
					}
				}
				timeZone.close();
				this.m_timeZone = (tz == null ? "Atlantic/Reykjavik" : tz);
			} catch (Exception e) {
				this.m_timeZone = "Europe/London";
			}
		}
		sdf.setTimeZone(TimeZone.getTimeZone(this.m_timeZone));
		return sdf.format(new Date());
	}
	/**
	 * Compute the authorization string.
	 * 
	 * @param restRequest		the URI
	 * @param rfcdate			the date in rfc format
	 * @return the authorization string
	 * @throws NoSuchAlgorithmException
	 */
	private String setAuth(String restRequest, String rfcdate) throws NoSuchAlgorithmException {
		String fullkey = (restRequest.substring(0, 1).matches("/")) ? restRequest.substring(1) : restRequest;
		fullkey += m_apiKey + rfcdate;
		MessageDigest md = MessageDigest.getInstance("MD5");
		md.update(fullkey.getBytes());
		byte byteData[] = md.digest();
		//convert the byte to hex format method 1
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < byteData.length; i++) {
			sb.append(Integer.toString((byteData[i] & 0xff) + 0x100, 16).substring(1));
		}
		return sb.toString();
	}
	/**
	 * Set up the common properties of the connection.
	 * 
	 * @param apiConn
	 * @param sb
	 * @param rfcdate
	 * @param value
	 * @throws IOException
	 */
	private void setUpConn(HttpURLConnection apiConn, String sb, String rfcdate, String value, String method)
			throws IOException {
		apiConn.setRequestMethod(method);
		apiConn.setRequestProperty("Accept", "application/json");
		apiConn.setRequestProperty("Authorization", "api-auth-" + m_apiKeyID + "-" + sb);
		if (method.equalsIgnoreCase("PUT") || method.equalsIgnoreCase("POST")) {
			apiConn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
		}
		apiConn.setRequestProperty("charset", "utf-8");
		apiConn.setRequestProperty("Date", rfcdate);
		apiConn.setRequestProperty("Content-Length", "" + Integer.toString(value.getBytes().length));
		apiConn.setRequestProperty("X-SkySQL-API-Version", "1.1");
		apiConn.setDoOutput(true);
		apiConn.setUseCaches(false);
		if ( (!method.equalsIgnoreCase("GET")) && value.length() > 1) {
			OutputStreamWriter out = new OutputStreamWriter(apiConn.getOutputStream());
			out.write(value);
			out.flush();
			out.close();
		}
		return;
	}
	/**
	 * Run the buffer queue.
	 */
	private void runBuffer() {
		if (this.m_bufferingExecution && (this.m_cycles >= 10)) {
			APIBatchExecution.sendAll();
			this.m_cycles = 0;
		}
		this.m_cycles++;
	}
	/**
	 * Save the failed request to the stack.
	 * 
	 * @param typeValue			the name of the method to call
	 * @param restRequest		the request URI
	 * @param pName				the name of the parameters to pass to the API
	 * @param pValue			the value of the parameters to pass to the API
	 */
	private void pushFailedApi(String typeValue, String restRequest, String[] pName, String[] pValue) {
		List<Object> toBatch = new ArrayList<Object>();
		toBatch.add(typeValue);
		toBatch.add(restRequest);
		toBatch.add(pName);
		toBatch.add(pValue);
		try {
			toBatch.add(setDate());
		} catch (IOException e) {
			// IGNORE
		}
		APIBatchExecution.push(toBatch);
	}










	/**
	 * "Singleton" to implement the API buffering mechanism.
	 * 
	 * @author Massimo Siani
	 *
	 */
	private static class APIBatchExecution {
		/**
		 * The "stack" contains the failed calls according to the following logic:
		 * 		(java method, request URI, name of parameters, value of parameters).
		 */
		private List<List<Object>> stack = new ArrayList<List<Object>>();
		/**
		 * Backup of the stack, for performing operations on the stack.
		 */
		private List<List<Object>> stack_bkp = new ArrayList<List<Object>>();
		/**
		 * Set a new MonAPI instance that will not call the batch queue
		 * 		(otherwise infinite loop).
		 */
		private MonAPI mapi = new MonAPI(false, 3);
		/**
		 * Instance of the present class. To be instantiate only once, thus
		 * keep it private.
		 */
		private static volatile APIBatchExecution INSTANCE = null;
		/**
		 * Constructor, to be kept private (only one stack may exist).
		 */
		private APIBatchExecution() {}
		/**
		 * Constructor wrapper, for double-checking the INSTANCE uniqueness.
		 */
		private synchronized static void APIBatchExecutionHolder() { 
			if (INSTANCE == null)
				INSTANCE = new APIBatchExecution();
		}
		/**
		 * Instantiates the unique instance of this class.
		 * 
		 * @return the instantiated INSTANCE
		 */
		private static APIBatchExecution getInstance() {
			if (INSTANCE == null)
				APIBatchExecutionHolder();
			return INSTANCE;
		}
		/**
		 * Transfers the stack into its backup, and clears the stack.
		 */
		private void backupBatchQueue() {
			getInstance().stack_bkp.addAll(getInstance().stack);
			getInstance().stack.clear();
		}
		/**
		 * Returns the whole stack list.
		 * @return
		 */
		//		public static List<List<Object>> getExecutionStack() {
		//			return getInstance().stack;
		//		}
		/**
		 * Pushes an element, on a LIFO.
		 * @param toBuffer
		 * @return true if succeeds or false on errors
		 */
		public synchronized static boolean push(List<Object> toBuffer) {
			try {
				getInstance().stack.add(toBuffer);
			} catch (Exception e) {
				return false;
			}
			return true;
		}
		/**
		 * Pops on a LIFO.
		 * @return
		 */
		//		public synchronized static List<Object> pop() {
		//			try {
		//				int lastPos = getInstance().stack.size() -1;
		//				List<Object> result = getInstance().stack.get(lastPos);
		//				getInstance().stack.remove(lastPos);
		//				return result;
		//			} catch (Exception e) {
		//				return null;
		//			}
		//		}
		/**
		 * Executes the buffer queue.
		 */
		public synchronized static void sendAll() {
			Method method;
			Class<? extends Object> classRequest;
			Class<? extends Object> classOne, classTwo;
			getInstance().backupBatchQueue();
			Logging.info("START BUFFERED REQUESTS: " + getInstance().stack_bkp.size() + " queued requests.");
			for (List<Object> batchCmd : getInstance().stack_bkp) {
				try {
					classRequest = batchCmd.get(1).getClass();
					classOne = batchCmd.get(2).getClass(); classOne =  String[].class;
					classTwo = batchCmd.get(3).getClass(); classTwo =  String[].class;
					method = getInstance().mapi.getClass().getDeclaredMethod((String) batchCmd.get(0),
							classRequest, classOne, classTwo);
					method.invoke(getInstance().mapi,
							batchCmd.get(1), batchCmd.get(2), batchCmd.get(3));
				} catch (ClassCastException e) {
					Logging.debug("Method " + (String) batchCmd.get(0)
							+ ": the parameters must be strings or extensions, got instead "
							+ batchCmd.get(1).getClass().getSimpleName() + ", " 
							+ batchCmd.get(2).getClass().getSimpleName() + ", "
							+ batchCmd.get(3).getClass().getSimpleName());
				} catch (NoSuchMethodException e) {
					Logging.debug("No method " + (String) batchCmd.get(0)
							+ " with parameters of type "
							+ batchCmd.get(1).getClass().getSimpleName() + ", "
							+ batchCmd.get(2).getClass().getSimpleName() + ", "
							+ batchCmd.get(3).getClass().getSimpleName());
				} catch (Exception e) {
					// ignore
				}
			}
			getInstance().stack_bkp.clear();
			Logging.info("STOP BUFFERED REQUESTS");
		}
	}
}