/*
 * Copyright (C) 2007-2017 Tatsuo Satoh <multisqllib@gmail.com>
 *
 * This file is part of sqlapp-gradle-plugin.
 *
 * sqlapp-gradle-plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sqlapp-gradle-plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with sqlapp-gradle-plugin.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sqlapp.gradle.plugins

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.tomcat.jdbc.pool.PoolConfiguration;
import org.apache.tomcat.jdbc.pool.PoolProperties;

import com.sqlapp.data.db.dialect.Dialect;
import com.sqlapp.data.db.dialect.DialectResolver;
import com.sqlapp.gradle.plugins.pojo.DataSourcePojo
import com.sqlapp.jdbc.JdbcUtils;
import com.sqlapp.jdbc.SqlappDataSource;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Optional;

abstract class AbstractDbTask extends AbstractTask{

	private Dialect dialect;

	protected Connection getConnection() {
		return this.getDataSource().getConnection();
	}

	protected PoolConfiguration getPoolConfiguration(DataSourcePojo obj) {
		PoolConfiguration poolConfiguration = new PoolProperties();
		poolConfiguration.setDriverClassName(getDriverClassName(obj.driverClassName, obj.url));
		poolConfiguration.setUrl(obj.url);
		poolConfiguration.setUsername(obj.username);
		poolConfiguration.setPassword(obj.password);
		poolConfiguration.setDefaultTransactionIsolation(obj.defaultTransactionIsolation);
		poolConfiguration.setInitialSize(obj.initialSize);
		return poolConfiguration;
	}

	protected DataSource newDataSource(def obj) {
		DataSource ds = null;
		if (obj instanceof DataSource){
			ds=obj;
		}
		if (!isDebug()) {
			ds = new org.apache.tomcat.jdbc.pool.DataSource(
					getPoolConfiguration(obj));
		} else {
			if (ds!=null){
				if (ds instanceof SqlappDataSource){
					SqlappDataSource sds =(SqlappDataSource)ds;
					sds.setDebug(isDebug());
					return ds;
				}
				SqlappDataSource sds =new SqlappDataSource(ds);
				sds.setDebug(isDebug());
				ds=sds;
			} else{
				if (!isDebug()) {
					ds = new org.apache.tomcat.jdbc.pool.DataSource(
								getPoolConfiguration(obj));
				} else{
					SqlappDataSource sds = new SqlappDataSource(
							new org.apache.tomcat.jdbc.pool.DataSource(
									getPoolConfiguration(obj)));
					sds.setDebug(isDebug());
					ds=sds;
				}
			}
		}
		return ds;
	}

	/**
	 * @return the driverClassName
	 */
	protected String getDriverClassName(String driverClassName, String url) {
		if (driverClassName == null) {
			driverClassName = JdbcUtils.getDriverClassNameByUrl(url);
		}
		return driverClassName;
	}
	
	/**
	 * @return the dataSource
	 */
	protected DataSource createDataSource() {
		if (this.pojo!=null){
			Object ds=pojo.dataSource;
			if (ds instanceof DataSource){
				return (DataSource)ds;
			} else{
				return newDataSource(ds);
			}
		}
		return null;
	}

	/**
	 * @return the dialect
	 */
	public Dialect getDialect() {
		if (this.dialect==null){
			Connection conn=this.getConnection();
			try{
				this.dialect=DialectResolver.getInstance().getDialect(conn);
			} finally{
				if (conn!=null){
					conn.close();
				}
			}
		}
		return dialect;
	}
}
