/*
 * Copyright 2014 Stackify
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.stackify.metric.impl;

import java.net.HttpURLConnection;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.AbstractScheduledService.CustomScheduler;
import com.stackify.api.common.http.HttpException;

/**
 * MetricBackgroundServiceScheduler
 * 
 * @author Eric Martin
 */
public class MetricBackgroundServiceScheduler extends CustomScheduler {
		
	/**
	 * Five seconds (milliseconds)
	 */
	private static final long FIVE_SECONDS = 5000;
		
	/**
	 * Fifteen seconds (milliseconds)
	 */
	private static final long FIFTEEN_SECONDS = 15000;
	
	/**
	 * Five minutes (milliseconds)
	 */
	private static final long FIVE_MINUTES = 300000;
	
	/**
	 * Schedule delay (in milliseconds)
	 */
	private long scheduleDelay = FIVE_SECONDS;
		
	/**
	 * Sets the next scheduled delay based on the number of metrics sent
	 * @param numSent The number of metrics sent
	 */
	public void update(final int numSent) {
		
		// adjust the schedule delay back to 5 seconds
		
		scheduleDelay = FIVE_SECONDS;
	}

	/**
	 * Sets the next scheduled delay based on the HTTP transmission status
	 * @param t The exception
	 */
	public void update(final Throwable t) {
		
		// see if the exception indicates an authorization problem
		
		boolean isAuthorized = true;
		
		if (t instanceof HttpException) {
			HttpException httpException = (HttpException) t;
			
			if (httpException.getStatusCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
				isAuthorized = false;
			}
		}
		
		// adjust the schedule delay based on the type of error 
		
		if (isAuthorized) {
			
			// Set the schedule delay to 15 seconds
			
			scheduleDelay = FIFTEEN_SECONDS;
			
		} else {
			
			// Set the schedule delay to five minutes
			
			scheduleDelay = FIVE_MINUTES;
		}
	}
	
	/**
	 * @return the scheduleDelay
	 */
	public long getScheduleDelay() {
		return scheduleDelay;
	}

	/**
	 * @see com.google.common.util.concurrent.AbstractScheduledService.CustomScheduler#getNextSchedule()
	 */
	@Override
	protected Schedule getNextSchedule() {
		return new Schedule(scheduleDelay, TimeUnit.MILLISECONDS);
	}
}
