/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils.pagination;

import java.net.http.HttpResponse;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.stackone.stackone_client_java.utils.ResponseWithBody;
import com.stackone.stackone_client_java.utils.Blob;
import com.stackone.stackone_client_java.utils.SpeakeasyLogger;

import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.ReadContext;
import com.jayway.jsonpath.spi.json.JacksonJsonProvider;
import com.jayway.jsonpath.spi.mapper.JacksonMappingProvider;

// Internal API only

/**
 * A non-blocking generic pagination implementation that handles fetching and publishing paginated data.
 * This class implements the {@link Flow.Publisher} interface for asynchronous streaming of paginated responses.
 * It uses a ProgressTrackerStrategy to process pagination metadata from responses and determine
 * when to stop pagination.
 * <p>
 * The pagination flow works as follows:
 * 1. subscribe() creates a new subscription for the subscriber
 * 2. The subscription fetches pages asynchronously using CompletableFuture
 * 3. Each response is processed by the progress tracker to update pagination state
 * 4. Pages are emitted to the subscriber until pagination is exhausted or cancelled
 *
 * @param <ReqT>           The type of the request object
 * @param <ProgressParamT> The type of the progression parameter (e.g., page number, offset, cursor)
 */
public class AsyncPaginator<ReqT, ProgressParamT> implements Flow.Publisher<HttpResponse<Blob>> {

    private static final SpeakeasyLogger logger = SpeakeasyLogger.getLogger(AsyncPaginator.class);
    
    /**
     * The initial request containing pagination parameters.
     */
    private final ReqT initialRequest;
    /**
     * The handler used to process pagination logic.
     */
    private final ProgressTrackerStrategy<ProgressParamT> progressTracker;
    /**
     * Function that sets the value for pagination.
     */
    private final BiFunction<ReqT, ProgressParamT, ReqT> requestModifier;
    /**
     * Function that fetches the next page of data asynchronously.
     */
    private final Function<ReqT, CompletableFuture<HttpResponse<Blob>>> asyncDataFetcher;

    private static final Configuration JSON_PATH_CONFIG = Configuration.defaultConfiguration()
            .jsonProvider(new JacksonJsonProvider())
            .mappingProvider(new JacksonMappingProvider())
            .addOptions(Option.SUPPRESS_EXCEPTIONS);

    /**
     * Creates a new AsyncPaginator instance.
     *
     * @param initialRequest     The initial request to use for the first page
     * @param progressTracker    The handler that processes pagination metadata from responses
     * @param requestModifier    Function that sets the pagination value in the request
     * @param asyncDataFetcher   Function that fetches the response for a given request asynchronously
     */
    public AsyncPaginator(ReqT initialRequest,
                         ProgressTrackerStrategy<ProgressParamT> progressTracker,
                         BiFunction<ReqT, ProgressParamT, ReqT> requestModifier,
                         Function<ReqT, CompletableFuture<HttpResponse<Blob>>> asyncDataFetcher) {
        this.initialRequest = initialRequest;
        this.progressTracker = progressTracker;
        this.requestModifier = requestModifier;
        this.asyncDataFetcher = asyncDataFetcher;
    }

    @Override
    public void subscribe(Flow.Subscriber<? super HttpResponse<Blob>> subscriber) {
        subscriber.onSubscribe(new AsyncPaginationSubscription(subscriber));
    }

    /**
     * Internal subscription implementation that handles the async pagination logic.
     */
    private class AsyncPaginationSubscription implements Flow.Subscription {
        private final Flow.Subscriber<? super HttpResponse<Blob>> subscriber;
        private final AtomicReference<PaginationState> state = new AtomicReference<>(PaginationState.INITIAL);
        private final AtomicBoolean cancelled = new AtomicBoolean(false);
        private final AtomicLong demand = new AtomicLong(0);

        public AsyncPaginationSubscription(Flow.Subscriber<? super HttpResponse<Blob>> subscriber) {
            this.subscriber = subscriber;
        }

        @Override
        public void request(long n) {
            if (n <= 0) {
                subscriber.onError(new IllegalArgumentException("Request count must be positive"));
                return;
            }

            long currentDemand = demand.addAndGet(n);
            if (currentDemand < 0) {
                demand.set(Long.MAX_VALUE);
            }

            fetchNextIfNeeded();
        }

        @Override
        public void cancel() {
            cancelled.set(true);
        }

        private void fetchNextIfNeeded() {
            if (cancelled.get()) return;
            
            if (demand.get() <= 0 || state.get() == PaginationState.EXHAUSTED) {
                return;
            }

            fetchNextPage()
                .thenAccept(this::handleResponse)
                .exceptionally(this::handleError);
        }

        private CompletableFuture<HttpResponse<Blob>> fetchNextPage() {
            PaginationState currentState = state.get();
            ProgressParamT currentValue = progressTracker.getPosition();

            if (currentState == PaginationState.INITIAL) {
                logger.debug("Async fetching initial page");
            } else {
                logger.debug("Async fetching next page with position: {}", currentValue);
            }

            ReqT request = currentState == PaginationState.INITIAL ?
                    initialRequest :
                    requestModifier.apply(initialRequest, currentValue);

            return asyncDataFetcher.apply(request);
        }

        private void handleResponse(HttpResponse<Blob> response) {
            if (cancelled.get()) return;

            try {
                // Convert Blob to byte array for JsonPath processing
                response.body().toByteArray()
                    .thenAccept(data -> {
                        try {
                            ReadContext respJson = JsonPath.using(JSON_PATH_CONFIG).parseUtf8(data);

                            boolean hasMorePages = progressTracker.advance(respJson);
                            state.set(hasMorePages ? PaginationState.HAS_MORE_PAGES : PaginationState.EXHAUSTED);

                            if (logger.isTraceEnabled()) {
                                logger.trace("Async page fetched - status: {}, hasMorePages: {}", response.statusCode(), hasMorePages);
                            }

                            // Create new Blob with the data for the subscriber
                            Blob newBody = Blob.from(data);
                            HttpResponse<Blob> newResponse = new ResponseWithBody<>(response, body -> newBody);

                            demand.decrementAndGet();
                            subscriber.onNext(newResponse);

                            if (state.get() == PaginationState.EXHAUSTED) {
                                logger.debug("Async pagination exhausted");
                                subscriber.onComplete();
                            } else {
                                fetchNextIfNeeded();
                            }
                        } catch (Exception e) {
                            handleError(e);
                        }
                    })
                    .exceptionally(this::handleError);
                    
            } catch (Exception e) {
                handleError(e);
            }
        }

        private Void handleError(Throwable error) {
            if (!cancelled.get()) {
                logger.debug("Async pagination error: {}", error.getMessage());
                subscriber.onError(error);
            }
            return null;
        }
    }


    /**
     * Enum representing the current state of pagination.
     */
    private enum PaginationState {
        /**
         * Initial state - no requests made yet
         */
        INITIAL,
        /**
         * More pages are available
         */
        HAS_MORE_PAGES,
        /**
         * No more pages available
         */
        EXHAUSTED
    }
}
