package com.stackspot.plugin.util

import com.stackspot.plugin.openapi.GenerateOpenApiTaskConfigurator.Framework
import java.util.SortedSet

class KotlinLang {

    data class KClass(
        val name: String,
        val modifier: KModifier? = null,
        val packageName: String? = null,
        val isInterface: Boolean? = false,
        val isData: Boolean? = false,
        val isEnum: Boolean? = false,
        val constructorArgs: List<KProperty>? = emptyList(),
        val extraProperties: List<Properties>? = emptyList(),
        val functions: List<KFunction>? = emptyList(),
        val isAnnotated: Boolean? = false,
        val annotations: List<KAnnotation>? = null,
        val framework: Framework? = null,
        val extraImports: List<String> = emptyList()
    ) {
        fun imports(): SortedSet<String> {

            Utils.addFrameworkImports(framework)

            val imports = mutableSetOf<String>()

            extraImports.forEach { it -> Utils.importMap[it]?.let { imports.add(it) } }

            this.annotations?.forEach { annotation -> annotation.toImport()?.let { imports.add(it) } }

            this.constructorArgs?.forEach { kProperty ->
                kProperty.toImport()?.let { imports.add(it) }
                kProperty.annotations?.forEach { annotation -> annotation.toImport()?.let { imports.add(it) } }
            }

            this.functions?.forEach { function ->
                function.annotations?.forEach { annotation ->
                    annotation.toImport()?.let {
                        imports.add(it)
                        annotation.extraProperties?.forEach { extraProperties ->
                            extraProperties.toImport()?.let { import -> imports.add(import) }
                        }
                    }
                }
                function.parameters?.forEach { kProperty ->
                    kProperty.toImport()?.let {
                        imports.add(it)
                    }
                    kProperty.annotations?.forEach { annotation ->
                        annotation.toImport()?.let {
                            imports.add(it)
                        }
                    }
                }
                function.returnType?.let {
                    Utils.importMap[it.substringAfter("<").substringBefore(">")]?.let { value ->
                        imports.add(value)
                    }
                }
            }
            return imports.toSortedSet()
        }
    }

    data class KFunction(
        val name: String,
        val modifier: KModifier? = null,
        val isSuspend: Boolean? = false,
        val returnType: String? = null,
        val parameters: List<KProperty>? = emptyList(),
        val responseType: String? = null,
        val isAnnotated: Boolean? = false,
        val annotations: List<KAnnotation>? = null,
    )

    data class KProperty(
        val name: String,
        val type: String,
        val isNullable: Boolean? = false,
        val isAnnotated: Boolean? = false,
        val defaultValue: String? = null,
        val annotations: List<KAnnotation>? = null,
        val hasOutType: Boolean? = false,
        val outType: String? = null
    ) {
        fun toImport() =
            if (this.hasOutType == true) Utils.importMap[this.outType!!] else Utils.importMap[this.type]
    }

    data class KAnnotation(
        val type: String,
        val defaultValue: String? = null,
        val hasExtraProperties: Boolean? = false,
        val extraProperties: List<Properties>? = null
    ) {
        fun toImport() = Utils.importMap[this.type]
    }

    data class Properties(val key: String, val value: Any, val hasComma: Boolean) {
        fun toImport() = Utils.importMap[this.key]
    }

    enum class KModifier(val value: String) {
        PRIVATE("private"),
        PROTECTED("protected"),
        INTERNAL("internal"),
        PUBLIC("public")
    }

    object Utils {

        fun addFrameworkImports(framework: Framework?) =
            when (framework) {
                Framework.SPRING -> importMapSpring.forEach { importMap[it.key] = it.value }
                Framework.MICRONAUT -> importMapMicronaut.forEach { importMap[it.key] = it.value }
                else -> {}
            }

        val importMap = mutableMapOf(
            /*"Boolean" to "kotlin.Boolean",
            "String" to "kotlin.String",
            "Int" to "kotlin.Int",
            "Float" to "kotlin.Float",
            "Long" to "kotlin.Long",
            "Double" to "kotlin.Double",*/
            "BigDecimal" to "java.math.BigDecimal",
            "LocalDate" to "java.time.LocalDate",
            "ZonedDateTime" to "java.time.ZonedDateTime",
            "File" to "java.io.File",
            "List" to "kotlin.collections.List",
            "List<Byte>" to "kotlin.collections.List",
            "Map" to "kotlin.collections.Map",
            "Map<String, Any?>" to "kotlin.collections.Map",
            "Timestamp" to "java.sql.Timestamp",
            "UUID" to "java.util.UUID",
            "Valid" to "javax.validation.Valid",
            "NotNull" to "javax.validation.constraints.NotNull",
            "NotBlank" to "javax.validation.constraints.NotBlank",
            "JsonFormat" to "com.fasterxml.jackson.annotation.JsonFormat",
        )
        private val importMapMicronaut = mutableMapOf(
            "Introspected" to "io.micronaut.core.annotation.Introspected",
            "Validated" to "io.micronaut.validation.Validated",
            "QueryValue" to "io.micronaut.http.annotation.QueryValue",
            "Header" to "io.micronaut.http.annotation.Header",
            "PathVariable" to "io.micronaut.http.annotation.PathVariable",
            "Produces" to "io.micronaut.http.annotation.Produces",
            "httpResponseForMicronaut" to "io.micronaut.http.HttpResponse",
            "Post" to "io.micronaut.http.annotation.Post",
            "Get" to "io.micronaut.http.annotation.Get",
            "Put" to "io.micronaut.http.annotation.Put",
            "Patch" to "io.micronaut.http.annotation.Patch",
            "Delete" to "io.micronaut.http.annotation.Delete",
            "Head" to "io.micronaut.http.annotation.Head",
            "Options" to "io.micronaut.http.annotation.Options",
            "Trace" to "io.micronaut.http.annotation.Trace",
            "Body" to "io.micronaut.http.annotation.Body",
        )

        private val importMapSpring = mutableMapOf(
            "PathVariable" to "org.springframework.web.bind.annotation.PathVariable",
            "httpResponseForSpring" to "java.net.http.HttpResponse",
            "RequestMapping" to "org.springframework.web.bind.annotation.RequestMapping",
            "RequestParam" to "org.springframework.web.bind.annotation.RequestParam",
            "PathVariable" to "org.springframework.web.bind.annotation.PathVariable",
            "RequestBody" to "org.springframework.web.bind.annotation.RequestBody",
            "method" to "org.springframework.web.bind.annotation.RequestMethod",
        )

        val typeMapping = mutableMapOf(
            "string" to "String",
            "boolean" to "Boolean",
            "integer" to "Int",
            "float" to "Float",
            "long" to "Long",
            "double" to "Double",
            "number" to "BigDecimal",
            "date" to "LocalDate",
            "dateTime" to "ZonedDateTime",
            "date-time" to "ZonedDateTime",
            "file" to "File",
            "array" to "List",
            "list" to "List",
            "map" to "Map",
            "uuid" to "UUID",
            "object" to "Map<String, Any?>",
            "binary" to "List<Byte>",
            "query" to "QueryValue",
            "email" to "String",
        )
    }
}
