package com.stackspot.plugin.util.extensions

import com.stackspot.plugin.util.KotlinLang
import io.swagger.v3.oas.models.OpenAPI
import io.swagger.v3.oas.models.Operation
import io.swagger.v3.oas.models.PathItem
import io.swagger.v3.oas.models.Paths
import io.swagger.v3.oas.models.media.ArraySchema
import io.swagger.v3.oas.models.media.Schema
import io.swagger.v3.oas.models.parameters.Parameter
import io.swagger.v3.parser.OpenAPIV3Parser
import io.swagger.v3.parser.util.RefUtils
import org.apache.commons.io.FilenameUtils
import java.io.File
import java.util.Locale
import java.util.StringTokenizer

fun PathItem.httpMethod(): List<Pair<PathItem.HttpMethod, Operation>> {
    val operations = mutableListOf<Pair<PathItem.HttpMethod, Operation>>()

    if (this.post != null) operations.add(PathItem.HttpMethod.POST to this.post)
    if (this.get != null) operations.add(PathItem.HttpMethod.GET to this.get)
    if (this.put != null) operations.add(PathItem.HttpMethod.PUT to this.put)
    if (this.patch != null) operations.add(PathItem.HttpMethod.PATCH to this.patch)
    if (this.delete != null) operations.add(PathItem.HttpMethod.DELETE to this.delete)
    if (this.head != null) operations.add(PathItem.HttpMethod.HEAD to this.head)
    if (this.options != null) operations.add(PathItem.HttpMethod.OPTIONS to this.options)
    if (this.trace != null) operations.add(PathItem.HttpMethod.TRACE to this.trace)

    return operations
}

fun Operation.returnType(): String? = kotlin.runCatching {
    this.responses.forEach { (status, response) ->
        if (status?.toInt() in 200..299) {
            response.content?.forEach { (_, value) ->
                if (value.schema?.type?.equals("array") == true) {
                    val name = RefUtils.computeDefinitionName((value.schema as ArraySchema).items.`$ref`)
                        .toCamelCaseCapitalized()
                    return "List<$name>"
                }
                return RefUtils.computeDefinitionName(value.schema.`$ref`).toCamelCaseCapitalized()
            }
        }
    }
    return null
}.getOrNull()

fun parameterTypeMapping(type: String, schema: Schema<Any>): String {

    val typeMapped = KotlinLang.Utils.typeMapping[type]!!

    if (typeMapped == "List") {
        val itemsType = (schema as ArraySchema).items.type

        itemsType.contains("#").let {

            val stringTokenizer = StringTokenizer(itemsType.reversed())
            val itemRefName = stringTokenizer.nextToken("/").reversed()
            return typeMapped.plus("<" + itemRefName.replaceFirstChar { it.uppercase() } + ">")
        }

        return typeMapped.plus("<" + itemsType.replaceFirstChar { it.uppercase() } + ">")
    }

    return typeMapped
}

fun Operation.toKpropertiesSpring(): List<KotlinLang.KProperty> {
    val properties = mutableListOf<KotlinLang.KProperty>()
    this.parameters?.forEach {
        val type = it.schema?.type
        if (type != null) {
            properties.add(
                KotlinLang.KProperty(
                    name = it.name.toCamelCase(),
                    type = parameterTypeMapping(type, it.schema),
                    annotations = it.getAnnotationsSpring(),
                    isNullable = !it.required
                )
            )
        }
    }
    this.requestBody?.content?.forEach {
        val bodyType = RefUtils.computeDefinitionName(it.value.schema.`$ref`)
        properties.add(
            KotlinLang.KProperty(
                name = bodyType.toCamelCase(),
                type = bodyType.toCamelCaseCapitalized(),
                annotations = listOf(
                    KotlinLang.KAnnotation(type = "Valid"),
                    KotlinLang.KAnnotation(type = "RequestBody")
                ),
            )
        )
    }

    return properties
}

fun Operation.toKpropertiesMicronaut(): List<KotlinLang.KProperty> {
    val properties = mutableListOf<KotlinLang.KProperty>()
    this.parameters?.forEach {
        val type = it.schema?.type
        if (type != null) properties.add(
            KotlinLang.KProperty(
                name = it.name.toCamelCase(),
                type = parameterTypeMapping(type, it.schema),
                annotations = it.getAnnotationsMicronaut(),
                isNullable = !it.required
            )
        )
    }
    this.requestBody?.content?.forEach {
        val bodyType = RefUtils.computeDefinitionName(it.value.schema.`$ref`)
        properties.add(
            KotlinLang.KProperty(
                name = bodyType.toCamelCase(),
                type = bodyType.toCamelCaseCapitalized(),
                annotations = listOf(KotlinLang.KAnnotation(type = "Valid"), KotlinLang.KAnnotation(type = "Body")),
            )
        )
    }

    return properties
}

fun Parameter.getAnnotationsMicronaut(): List<KotlinLang.KAnnotation>? {
    return when (this.`in`) {
        "query" -> listOf(KotlinLang.KAnnotation(type = "QueryValue"))
        "header" -> listOf(KotlinLang.KAnnotation(type = "Header"))
        "path" -> listOf(KotlinLang.KAnnotation(type = "PathVariable"))
        else -> null
    }
}

fun Parameter.getAnnotationsSpring(): List<KotlinLang.KAnnotation>? {
    return when (this.`in`) {
        "query" -> listOf(KotlinLang.KAnnotation(type = "RequestParam"))
        "header" -> listOf(KotlinLang.KAnnotation(type = "RequestHeader"))
        "path" -> listOf(KotlinLang.KAnnotation(type = "PathVariable"))
        else -> null
    }
}

fun Paths.toKFunctionsWithMicronaut(): List<KotlinLang.KFunction> {
    val functions = mutableListOf<KotlinLang.KFunction>()

    this.forEach { (path, pathItem) ->
        pathItem.httpMethod().forEach { (method, operation) ->
            functions.add(
                KotlinLang.KFunction(
                    name = operation.operationId.toLowerCamelCase(),
                    isAnnotated = true,
                    returnType = operation.returnType(),
                    annotations = httpMethodAnnotationsWithMicronaut(path, method, operation),
                    parameters = operation.toKpropertiesMicronaut()
                )
            )
        }
    }

    return functions
}

fun Paths.toKFunctionsWithSpring(): List<KotlinLang.KFunction> {
    val functions = mutableListOf<KotlinLang.KFunction>()

    this.forEach { (path, pathItem) ->
        pathItem.httpMethod().forEach { (method, operation) ->
            functions.add(
                KotlinLang.KFunction(
                    name = operation.operationId.toLowerCamelCase(),
                    isAnnotated = true,
                    returnType = operation.returnType(),
                    annotations = httpMethodAnnotationsWithSpring(path, method, operation),
                    parameters = operation.toKpropertiesSpring()
                )
            )
        }
    }

    return functions
}

private fun httpMethodAnnotationsWithMicronaut(
    path: String,
    method: PathItem.HttpMethod,
    operation: Operation
): List<KotlinLang.KAnnotation>? = kotlin.runCatching {
    mutableListOf<KotlinLang.KAnnotation>().apply {
        add(
            KotlinLang.KAnnotation(
                type = method.toString().lowercase().toCamelCaseCapitalized(),
                defaultValue = path,
            )
        )
        operation.responses?.forEach { (status, response) ->
            if (status?.toInt() in 200..299) {
                response.content?.forEach { (contentType, _) ->
                    add(
                        KotlinLang.KAnnotation(
                            type = "Produces",
                            defaultValue = contentType
                        )
                    )
                }
            }
        }
    }
}.getOrNull()

private fun httpMethodAnnotationsWithSpring(
    path: String,
    method: PathItem.HttpMethod,
    operation: Operation
): List<KotlinLang.KAnnotation>? = kotlin.runCatching {
    mutableListOf<KotlinLang.KAnnotation>().apply {

        val extraProperties = mutableListOf(
            KotlinLang.Properties("value", "[\"$path\"]", false),
            KotlinLang.Properties("method", "[RequestMethod." + method.toString().uppercase() + "]", false),
        )

        operation.responses?.forEach { (status, response) ->
            if (status?.toInt() in 200..299) {
                response.content?.forEach { (contentType, _) ->
                    extraProperties.add(
                        KotlinLang.Properties("produces", "[\"$contentType\"]", false)
                    )
                }
            }
        }

        add(
            KotlinLang.KAnnotation(
                type = "RequestMapping",
                defaultValue = null,
                true,
                extraProperties = extraProperties
            )
        )
    }
}.getOrNull()

/*fun Paths.toPathMap(): Map<String, List<Pair<String, PathItem>>> {
    val mapPaths = mutableMapOf<String, MutableList<Pair<String, PathItem>>>()

    this.forEach {
        val key = it.key.split("/")[1]
        var list = mapPaths.get(key)
        if (list == null) list = mutableListOf()
        list.add((it.key to it.value))
        mapPaths[key] = list
    }

    return mapPaths
}*/

fun String.listOpenApiSpecs(): List<Pair<String, OpenAPI>> =
    File(this).walk(FileWalkDirection.TOP_DOWN).filter { file ->
        file.isFile && (file.extension == "yaml" || file.extension == "yml" || file.extension == "json")
    }.toList().map { FilenameUtils.removeExtension(it.name) to OpenAPIV3Parser().read(it.path) }

fun String.mkdirs() {
    File(this).mkdirs()
}

fun String.toCamelCaseCapitalized(): String = this.replaceFirstChar { it.uppercase() }

fun String.toCamelCase(): String = this.replaceFirstChar { it.lowercase() }

fun String.toKebabCase() = replace("(?<=.)(?=\\p{Upper})".toRegex(), "-").lowercase(Locale.getDefault())

fun String.toLowerCamelCase(): String {
    val regex = "-[a-zA-Z]".toRegex()
    return regex.replace(this) {
        it.value.replace("-", "").replaceFirstChar { firstChar -> firstChar.uppercase() }
    }
}
