/*
 * Decompiled with CFR 0.152.
 */
package com.webull.openapi.core.http.initializer.token;

import com.webull.openapi.core.execption.ClientException;
import com.webull.openapi.core.logger.Logger;
import com.webull.openapi.core.logger.LoggerFactory;
import com.webull.openapi.core.utils.StringUtils;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

public class TokenStorage {
    private static final Logger logger = LoggerFactory.getLogger(TokenStorage.class);
    private static final String DEFAULT_TOKEN_PATH = "conf";
    private static final String DEFAULT_TOKEN_FILE = "token.txt";
    private static final String DEFAULT_ENV_TOKEN_DIR = "WEBULL_OPENAPI_TOKEN_DIR";
    private static final Set<Character> WINDOWS_INVALID_CHARS = Collections.unmodifiableSet(new HashSet<Character>(Arrays.asList(Character.valueOf('|'), Character.valueOf('?'), Character.valueOf('*'), Character.valueOf('<'), Character.valueOf('>'), Character.valueOf('\"'))));
    private static final Set<Character> POSIX_INVALID_CHARS = Collections.unmodifiableSet(new HashSet<Character>(Collections.singletonList(Character.valueOf('\u0000'))));
    private final Path storageTokenDir;
    private final Path tokenFile;

    public TokenStorage(String customTokenDir) {
        this.storageTokenDir = this.resolveDir(customTokenDir);
        this.validatePathAccess(this.storageTokenDir);
        this.ensureDirExists();
        this.tokenFile = this.storageTokenDir.resolve(DEFAULT_TOKEN_FILE);
        logger.info("TokenStorage initialized path:{}.", (Object)this.tokenFile);
        this.checkFileExists();
    }

    private Path resolveDir(String customTokenDir) {
        String rawDir;
        if (StringUtils.isNotBlank(customTokenDir)) {
            rawDir = customTokenDir.trim();
            logger.info("TokenStorage uses the custom configuration, token_storage_dir:{}.", (Object)rawDir);
        } else {
            rawDir = System.getenv(DEFAULT_ENV_TOKEN_DIR);
            if (rawDir != null && StringUtils.isNotBlank(rawDir)) {
                rawDir = rawDir.trim();
                logger.info("TokenStorage uses environment variable configuration, {}:{}.", DEFAULT_ENV_TOKEN_DIR, rawDir);
            } else {
                rawDir = System.getProperty("user.dir") + FileSystems.getDefault().getSeparator() + DEFAULT_TOKEN_PATH;
                logger.info("TokenStorage uses the default configuration, {}.", (Object)rawDir);
            }
        }
        String resolvedDir = rawDir.replace("~", System.getProperty("user.dir"));
        this.validatePathSyntax(resolvedDir);
        try {
            return Paths.get(resolvedDir, new String[0]).toAbsolutePath().normalize();
        }
        catch (Exception e) {
            logger.error("TokenStorage failed to get path, path:{}", resolvedDir, e);
            throw new ClientException("StorageTokenError", e);
        }
    }

    private void validatePathSyntax(String resolvedDir) {
        String os = System.getProperty("os.name").toLowerCase();
        Set<Character> invalidChars = os.contains("windows") ? WINDOWS_INVALID_CHARS : POSIX_INVALID_CHARS;
        for (char c : resolvedDir.toCharArray()) {
            if (!invalidChars.contains(Character.valueOf(c))) continue;
            String msg = String.format("TokenStorage path contains illegal characters '%c', path:%s.", Character.valueOf(c), resolvedDir);
            logger.warn(msg);
            break;
        }
    }

    private void validatePathAccess(Path path) {
        if (Files.exists(path, new LinkOption[0])) {
            if (!Files.isDirectory(path, new LinkOption[0])) {
                String msg = String.format("TokenStorage path already exists but is not a directory, path:%s.", path);
                logger.warn(msg);
                return;
            }
            if (!Files.isReadable(path) || !Files.isWritable(path)) {
                String msg = String.format("TokenStorage directory has no read/write permission. Please check the configuration, path:%s.", path);
                logger.warn(msg);
            }
        } else {
            Path parentDir = path.getParent();
            if (parentDir == null) {
                String msg = String.format("TokenStorage has no valid parent directory for the path, path:%s.", path);
                logger.warn(msg);
                return;
            }
            if (!Files.exists(parentDir, new LinkOption[0])) {
                String msg = String.format("TokenStorage parent directory does not exist, unable to create directory. Please check the configuration, parent path:%s.", parentDir.toAbsolutePath());
                logger.warn(msg);
                return;
            }
            if (!(Files.isReadable(parentDir) && Files.isWritable(parentDir) && Files.isExecutable(parentDir))) {
                String msg = String.format("TokenStorage parent directory has no read/write permission. Please check the configuration, parent path:%s.", parentDir.toAbsolutePath());
                logger.warn(msg);
            }
        }
    }

    private void ensureDirExists() {
        if (!Files.exists(this.storageTokenDir, new LinkOption[0])) {
            try {
                Files.createDirectories(this.storageTokenDir, new FileAttribute[0]);
            }
            catch (IOException e) {
                logger.error("TokenStorage failed to create directory, file:{}", this.storageTokenDir, e);
                throw new ClientException("StorageTokenError", e);
            }
        }
    }

    private void checkFileExists() {
        if (Files.exists(this.tokenFile, new LinkOption[0])) {
            String msg = String.format("TokenStorage Note: The token file already exists, the latest token configuration will be overwritten and written to the token file after successful 2FA verification. path:%s.", this.tokenFile);
            logger.warn(msg);
        }
    }

    public Path getTokenFile() {
        return this.tokenFile;
    }
}

