package dev.coderoutine.tabulate

import dev.coderoutine.tabulate.Align.LEFT
import dev.coderoutine.tabulate.Align.RIGHT
import dev.coderoutine.tabulate.TableScope.RowIteratorState.*


class TableScope<T>(
    /**
     * How to align the values in the cells.
     */
    var align: Align = LEFT,

    /**
     * Whether to print the headers.
     */
    var withHeader: Boolean = true,

    /**
     * How to align the headers.
     */
    var alignHeader: Align = align,

    /**
     * The character that should be used to fill empty positions when aligning values in cells.
     */
    var padWith: String = " ",

    /**
     * How many positions should be padded between cell values and the column separators.
     * This value only applies horizontally.
     */
    var paddingWidth: Int = 1,

    /**
     * Global value provider as replacement for `null` values.
     * By default, `null` is used as is.
     */
    var onNull: () -> Any? = { null }
) {
    private var lineCharset: LineCharset = LineCharset.Simple
    private var includeColumnSeparator = true
    private var includeHeaderSeparator = false
    private var includeLeftRightOutline = false
    private var includeTopBottomOutline = false

    private var headerFormatter: (String) -> Any = { it }
    private var cellFormatter: (T, String) -> Any = { _, it -> it }
    private val columns = mutableListOf<ColumnSpec<T>>()
    private lateinit var data: Data<T>

    /**
     * A function that applies a formatting to the header of this column.
     * The return value of this function must not change the visible width of the header,
     * because the formatting will be applied *after* the column measurement.
     *
     * This function is intended to be used for ANSI-formatting.
     *
     * If you want to apply special formatting that does affect visible length of the header,
     * do that beforehand and pass the final value as header right away.
     */
    fun formatHeader(formatter: (String) -> Any) {
        headerFormatter = formatter
    }

    /**
     * A function used for formatting cell values.
     * The returned value must not change the visible width of the value.
     * It is mainly intended to be used for ANSI-formatting.
     *
     * First argument is the current record.
     *
     * Second argument is the string-converted cell value of this record.
     */
    fun formatCell(formatter: (T, String) -> Any) {
        cellFormatter = formatter
    }

    /**
     * Define which lines to print and which characters to use for it.
     */
    fun lines(
        charset: LineCharset = this.lineCharset,
        betweenColumns: Boolean = this.includeColumnSeparator,
        afterHeader: Boolean = this.includeHeaderSeparator,
        leftRightOutline: Boolean = this.includeLeftRightOutline,
        topBottomOutline: Boolean = this.includeTopBottomOutline,
    ) {
        this.lineCharset = charset
        this.includeColumnSeparator = betweenColumns
        this.includeHeaderSeparator = afterHeader
        this.includeLeftRightOutline = leftRightOutline
        this.includeTopBottomOutline = topBottomOutline
    }

    /**
     * Add a column to the table with the given properties.
     */
    fun column(
        /**
         * The header of this column.
         */
        header: String = "",
        /**
         * A custom alignment for the cells that only applies to this column.
         * Defaults to [TableScope.align].
         */
        align: Align = this.align,
        /**
         * A custom alignment for the header that only applies to this column.
         * Defaults to [TableScope.alignHeader].
         */
        alignHeader: Align = this.alignHeader,
        /**
         * The character that should be used to fill empty positions when aligning values in cells of this column.
         * Defaults to [TableScope.paddingWidth].
         */
        padWith: String = this.padWith,
        /**
         * A fixed width for this column or `-1` if the column should be measured.
         * Is `-1` by default.
         */
        width: Int = -1,
        /**
         * Minimum width of this column.
         */
        minWidth: Int = 0,
        /**
         * Maximum width of this column.
         */
        maxWidth: Int = Int.MAX_VALUE,
        /**
         * A special function used for formatting values in this column.
         * The returned value must not change the visible width of the value.
         * It is mainly intended to be used for ANSI-formatting.
         *
         * Defaults to [TableScope.cellFormatter].
         */
        format: (record: T, cellValue: String) -> Any = this.cellFormatter,
        /**
         * Column specific value provider as replacement for `null` values.
         * By default, [TableScope.onNull] is used.
         */
        onNull: () -> Any? = this.onNull,
        /**
         * A function that extracts the value of this column from each instance of [T].
         * This function will be called once for each [T] in the given data.
         */
        valueProvider: (T) -> Any?,
    ) {
        columns.add(
            ColumnSpec(
                header = header,
                align = align,
                alignHeader = alignHeader,
                padWith = padWith,
                width = width,
                minWidth = minWidth,
                maxWidth = maxWidth,
                format = format,
                onNull = onNull,
                valueProvider = valueProvider,
            )
        )
    }

    internal fun measureColumns(data: Data<T>) {
        this.data = data

        mainLoop@ for (column in columns) {
            if (column.width >= 0) {
                continue
            }

            if (column.updateWidthWithinConstraints(column.header.length)) {
                // Skip iterating rest of data because maxWidth is already reached.
                continue
            }

            for (record in data.measurementIterator()) {
                val value = column.valueProvider(record) ?: column.onNull()
                val valueAsString = value.toString()

                // Cache valueAsString to avoid re-computation.
                column.values.add(valueAsString)

                if (column.updateWidthWithinConstraints(valueAsString.length)) {
                    // Skip iterating rest of data because maxWidth is already reached.
                    continue@mainLoop
                }
            }
        }
    }

    private fun StringBuilder.appendN(n: Int, char: String) {
        repeat(n) { append(char) }
    }

    private fun <T> StringBuilder.appendFormattedAndAligned(
        record: T,
        cellValue: String,
        columnWidth: Int,
        align: Align,
        paddingChar: String,
        formatter: (T, String) -> Any,
    ) {
        var clipped = cellValue

        // Clip value to width of column if necessary.
        // This must be done before formatting to avoid breaking ANSI control sequences.
        if (clipped.length > columnWidth) {
            clipped = clipped.substring(0, columnWidth)
        }

        val formatted = formatter(record, clipped)

        if (clipped.length == columnWidth) {
            // No padding necessary.
            append(formatted)
            return
        }

        val paddingWidth = columnWidth - clipped.length

        when (align) {
            LEFT -> {
                append(formatted)
                repeat(paddingWidth) { append(paddingChar) }
            }

            RIGHT -> {
                repeat(paddingWidth) { append(paddingChar) }
                append(formatted)
            }
        }
    }

    private inline fun StringBuilder.appendRow(
        leftEnd: String,
        padding: String,
        columnSeparator: String,
        rightEnd: String,
        fill: (idx: Int, ColumnSpec<T>) -> Unit
    ) {
        if (includeLeftRightOutline) {
            append(leftEnd)
        }

        for (idx in columns.indices) {
            val column = columns[idx]

            appendN(paddingWidth, padding)

            fill(idx, column)

            appendN(paddingWidth, padding)

            if (idx == columns.indices.last) {
                if (includeLeftRightOutline) {
                    append(rightEnd)
                }
            } else {
                if (includeColumnSeparator) {
                    append(columnSeparator)
                }
            }
        }
    }

    private fun StringBuilder.appendSeparatorRow(
        leftEnd: String,
        padding: String,
        columnSeparator: String,
        rightEnd: String,
        fillWith: String,
    ) = appendRow(
        leftEnd = leftEnd,
        padding = padding,
        columnSeparator = columnSeparator,
        rightEnd = rightEnd,
        fill = { _, column -> repeat(column.width) { append(fillWith) } }
    )

    private inline fun StringBuilder.appendValueRow(
        fill: (idx: Int, ColumnSpec<T>) -> Unit
    ) = appendRow(
        leftEnd = lineCharset.verticalOutline,
        padding = padWith,
        columnSeparator = lineCharset.verticalLine,
        rightEnd = lineCharset.verticalOutline,
        fill = fill,
    )

    private fun StringBuilder.appendTopLine() = appendSeparatorRow(
        leftEnd = lineCharset.topLeftCorner,
        padding = lineCharset.horizontalOutline,
        columnSeparator = lineCharset.topEnd,
        rightEnd = lineCharset.topRightCorner,
        fillWith = lineCharset.horizontalOutline,
    )

    private fun StringBuilder.appendHeaderRow() = appendValueRow { _, column ->
        appendFormattedAndAligned(
            Unit,
            column.header,
            column.width,
            column.alignHeader,
            column.padWith
        ) { _, it -> headerFormatter(it) }
    }

    private fun StringBuilder.appendHeaderSeparator() = appendSeparatorRow(
        leftEnd = lineCharset.leftEnd,
        padding = lineCharset.horizontalLine,
        columnSeparator = lineCharset.intersection,
        rightEnd = lineCharset.rightEnd,
        fillWith = lineCharset.horizontalLine,
    )

    private fun StringBuilder.appendBodyRow(rowIdx: Int, rowValue: T) = appendValueRow { _, column ->
        val cellValue = if (column.values.size <= rowIdx) {
            // Recalculate string value.
            column.valueProvider(rowValue)?.toString() ?: column.onNull().toString()
        } else {
            column.values[rowIdx]
        }

        appendFormattedAndAligned(
            rowValue,
            cellValue,
            column.width,
            column.align,
            column.padWith,
            column.format
        )
    }

    private fun StringBuilder.appendBottomLine() = appendSeparatorRow(
        leftEnd = lineCharset.bottomLeftCorner,
        padding = lineCharset.horizontalOutline,
        columnSeparator = lineCharset.bottomEnd,
        rightEnd = lineCharset.bottomRightCorner,
        fillWith = lineCharset.horizontalOutline,
    )

    internal fun rowIterator(): Iterator<String> = RowIterator(data.outputIterator())

    private inner class RowIterator(
        private val iterator: Iterator<T>,
    ) : Iterator<String> {
        private val sb = StringBuilder()

        private var rowIdx = -1
        private var bottomLineAppended = false
        private var state = when {
            includeTopBottomOutline -> TOP_LINE
            withHeader -> HEADER
            else -> BODY
        }

        override fun hasNext(): Boolean {
            return when {
                state == TOP_LINE && includeTopBottomOutline -> true
                state == HEADER && withHeader -> true
                state == HEADER_SEPARATOR && includeColumnSeparator -> true
                state == BODY && iterator.hasNext() -> true
                else -> {
                    // if iterator.hasNext == false, snap to bottom line always.
                    // Even though this mutates the state, hasNext is still idempotent,
                    // because the actual return value is based on bottomLineAppended.
                    state = BOTTOM_LINE
                    includeTopBottomOutline && !bottomLineAppended
                }
            }
        }

        override fun next(): String {
            sb.clear()

            when (state) {
                TOP_LINE -> {
                    state = if (withHeader) HEADER else BODY
                    sb.appendTopLine()
                }

                HEADER -> {
                    state = if (includeHeaderSeparator) HEADER_SEPARATOR else BODY
                    sb.appendHeaderRow()
                }

                HEADER_SEPARATOR -> {
                    state = BODY
                    sb.appendHeaderSeparator()
                }

                BODY -> {
                    sb.appendBodyRow(++rowIdx, iterator.next())
                }

                BOTTOM_LINE -> {
                    bottomLineAppended = true
                    sb.appendBottomLine()
                }
            }

            return sb.toString()
        }
    }

    private enum class RowIteratorState {
        TOP_LINE, HEADER, HEADER_SEPARATOR, BODY, BOTTOM_LINE
    }
}