package dev.coderoutine.tabulate

import kotlin.math.max

interface ColumnScope {
    /**
     * Can be used to apply formatting on the given cell value.
     * This function is called just before printing the line containing this value.
     * The [String] argument given to the [format] function can be one of the following:
     *
     * - The result of [toString] of the receiver object of this function.
     * - The result of [toString] of the receiver object of this function clipped to the width of the column.
     * - The result of [toString] of the value returned by the columns `onNull` provider, if the receiver object is `null`.
     *
     * Because this function is called after the layout phase and
     * after the cell values have been converted to strings and clipped,
     * the value returned by [format] *must not* change the visual width of
     * the value given to it. Otherwise, the table layout will break!
     *
     * Changing the casing style or using ANSI escape sequences is just fine,
     * because they keep the visual width of the value.
     */
    infix fun Any?.formatted(format: (String) -> Any): FormattedValue
}

internal class ColumnSpec<T>(
    val header: String,
    val align: Align,
    val alignHeader: Align,
    val padWith: String,
    var width: Int,
    var minWidth: Int,
    var maxWidth: Int,
    var onNull: () -> Any?,
    var footer: Any,
    val valueProvider: ColumnScope.(T) -> Any?,
) : ColumnScope {
    val values = mutableListOf<Any?>()

    override infix fun Any?.formatted(format: (String) -> Any): FormattedValue {
        return FormattedValue(this, this?.toString() ?: onNull().toString(), format)
    }

    /**
     * Updates the width of this column and makes sure that [minWidth]
     * and [maxWidth] are taken into account.
     *
     * @return `true` if [maxWidth] was reached, allowing an early quit of measurement, `false` otherwise.
     */
    fun updateWidthWithinConstraints(value: Int): Boolean {
        width = max(width, max(minWidth, value))

        if (width >= maxWidth) {
            width = maxWidth
            return true
        }

        return false
    }
}