package dev.coderoutine.tabulate

internal const val ANSI_ESC = "\u001b["
internal const val ANSI_RESET = "${ANSI_ESC}0m"
internal val ANSI_ESC_REGEX = Regex("""\x1B([@-Z\\-_]|\[[0-?]*[ -/]*[@-~])?""")

/**
 * Dedicated function for measuring the total length of all matches of this [Regex]
 * within [text]. Avoids creating intermediate [String] objects or collections.
 */
private fun Regex.totalLengthOfMatches(text: String): Int {
    val matcher = this.toPattern().matcher(text)
    if (!matcher.find()) return 0 // No matches, return zero.

    var lengthOfMatchingParts = 0

    do {
        lengthOfMatchingParts += (matcher.end() - matcher.start())
    } while (matcher.find())

    return lengthOfMatchingParts
}

internal val String.lengthWithoutAnsiEscapeChars get() = length - ANSI_ESC_REGEX.totalLengthOfMatches(this)

internal fun appendAnsiEscapeAwareClippedTo(
    sb: StringBuilder,
    text: String,
    maxWidth: Int,
) {
    if (text.length <= maxWidth) {
        sb.append(text)
        return
    }

    val matcher = ANSI_ESC_REGEX.toPattern().matcher(text)

    if (!matcher.find()) {
        sb.append(text.substring(0, maxWidth))
        return
    }

    var currentIdx = 0
    var appendedLength = 0
    var startOfEscSeq: Int
    var endOfEscSeq: Int
    var escapeSeqAdded = false
    var quitEarly = false

    do {
        startOfEscSeq = matcher.start()
        endOfEscSeq = matcher.end()

        if (startOfEscSeq > currentIdx) {
            // There is text before the next escape sequence. Add that one first.
            val lengthOfPartBeforeSeq = startOfEscSeq - currentIdx
            val lengthToUse = lengthOfPartBeforeSeq.coerceAtMost(maxWidth - appendedLength)
            sb.append(text.substring(currentIdx, currentIdx + lengthToUse))
            appendedLength += lengthToUse

            if (appendedLength >= maxWidth) {
                // Quit early. An additional ANSI RESET will be appended anyway,
                // so no need to include the subsequent escape sequence too.
                quitEarly = true
                break
            }
        }

        // Append escape sequence.
        sb.append(text.substring(startOfEscSeq, endOfEscSeq))
        escapeSeqAdded = true

        currentIdx = matcher.end()
    } while (matcher.find())

    if (endOfEscSeq < text.length && appendedLength < maxWidth) {
        // There is left over text behind the last escape sequence.
        val lengthToUse = (text.length - endOfEscSeq).coerceAtMost(maxWidth - appendedLength)
        sb.append(text.substring(endOfEscSeq, endOfEscSeq + lengthToUse))
    }

    if (escapeSeqAdded && quitEarly) {
        sb.append(ANSI_RESET)
    }
}



