package dev.coderoutine.tabulate

import dev.coderoutine.tabulate.TableLinesIterator.State.*

internal class TableLinesIterator<T>(
    private val table: TableSpec<T>,
    private val iterator: Iterator<T>,
) : Iterator<String> {
    private val sb = StringBuilder()

    private var rowIdx = -1
    private var bottomLineAppended = false
    private var lastRowAppended = false
    private var state = when {
        table.includeTopBottomOutline -> TOP_LINE
        table.withHeader -> HEADER
        else -> BODY
    }

    override fun hasNext(): Boolean {
        return when {
            state == TOP_LINE && table.includeTopBottomOutline -> true
            state == HEADER && table.withHeader -> true
            state == HEADER_SEPARATOR && table.includeColumnSeparator -> true
            state == BODY && iterator.hasNext() -> true
            state == ROW_SEPARATOR -> true
            state == FOOTER && table.withFooter -> true
            else -> {
                // if iterator.hasNext == false, snap to bottom line always.
                // Even though this mutates the state, hasNext is still idempotent,
                // because the actual return value is based on bottomLineAppended.
                state = BOTTOM_LINE
                table.includeTopBottomOutline && !bottomLineAppended
            }
        }
    }

    override fun next(): String {
        sb.clear()

        when (state) {
            TOP_LINE -> {
                state = if (table.withHeader) HEADER else BODY
                table.appendTopLineTo(sb)
            }

            HEADER -> {
                state = if (table.includeHeaderSeparator) HEADER_SEPARATOR else BODY
                table.appendHeaderRowTo(sb)
            }

            HEADER_SEPARATOR -> {
                state = BODY
                table.appendHeaderSeparatorTo(sb)
            }

            BODY -> {
                table.appendBodyRowTo(sb, ++rowIdx, iterator.next())
                lastRowAppended = !iterator.hasNext()

                if (table.rowSeparatorPredicate(rowIdx, lastRowAppended)) {
                    state = ROW_SEPARATOR
                } else if (lastRowAppended) {
                    state = FOOTER
                }
            }

            ROW_SEPARATOR -> {
                table.appendRowSeparatorTo(sb)
                state = if (lastRowAppended) FOOTER else BODY
            }

            FOOTER -> {
                table.appendFooterRowTo(sb)
                state = BOTTOM_LINE
            }

            BOTTOM_LINE -> {
                bottomLineAppended = true
                table.appendBottomLineTo(sb)
            }
        }

        return sb.toString()
    }

    private enum class State {
        TOP_LINE, HEADER, HEADER_SEPARATOR, BODY, BOTTOM_LINE, ROW_SEPARATOR, FOOTER
    }
}
