package dev.coderoutine.tabulate

private inline fun <T> TableSpec<T>.appendRowTo(
    sb: StringBuilder,
    leftEnd: String,
    padding: String,
    columnSeparator: String,
    rightEnd: String,
    fill: (idx: Int, ColumnSpec<T>) -> Unit
) {
    if (includeLeftRightOutline) {
        sb.append(leftEnd)
    }

    for (idx in columns.indices) {
        val column = columns[idx]

        sb.appendN(paddingWidth, padding)

        fill(idx, column)

        sb.appendN(paddingWidth, padding)

        if (idx == columns.indices.last) {
            if (includeLeftRightOutline) {
                sb.append(rightEnd)
            }
        } else {
            if (includeColumnSeparator) {
                sb.append(columnSeparator)
            }
        }
    }
}

private fun <T> TableSpec<T>.appendSeparatorRowTo(
    sb: StringBuilder,
    leftEnd: String,
    padding: String,
    columnSeparator: String,
    rightEnd: String,
    fillWith: String,
) = appendRowTo(
    sb = sb,
    leftEnd = leftEnd,
    padding = padding,
    columnSeparator = columnSeparator,
    rightEnd = rightEnd,
    fill = { _, column -> repeat(column.width) { sb.append(fillWith) } }
)

private inline fun <T> TableSpec<T>.appendValueRowTo(
    sb: StringBuilder,
    fill: (idx: Int, ColumnSpec<T>) -> Unit
) = appendRowTo(
    sb = sb,
    leftEnd = lineCharset.verticalOutline,
    padding = padWith,
    columnSeparator = lineCharset.verticalLine,
    rightEnd = lineCharset.verticalOutline,
    fill = fill,
)

private fun appendClippedOrAlignedIgnoringAnsiEsq(
    sb: StringBuilder,
    originalValue: Any?,
    valueAsString: String,
    columnWidth: Int,
    align: Align,
    paddingChar: String,
) {
    val paddingWidth = if (valueAsString.length < columnWidth) columnWidth - valueAsString.length else 0

    when (paddingWidth) {
        // No alignment needed. Check for clipping though.
        0 -> if (valueAsString.length > columnWidth) {
            sb.append(valueAsString.substring(0, columnWidth))
        } else {
            sb.append(valueAsString)
        }

        else -> sb.appendAligned(align, paddingWidth, paddingChar, valueAsString, originalValue)
    }
}

private fun appendClippedOrAlignedHandlingAnsiEsq(
    sb: StringBuilder,
    originalValue: Any?,
    valueAsString: String,
    columnWidth: Int,
    align: Align,
    paddingChar: String,
) {
    val length = valueAsString.lengthWithoutAnsiEscapeChars
    val paddingWidth = if (length < columnWidth) columnWidth - length else 0

    when (paddingWidth) {
        // No alignment needed. Clip ANSI aware...
        0 -> appendAnsiEscapeAwareClippedTo(sb, valueAsString, columnWidth)

        else -> sb.appendAligned(align, paddingWidth, paddingChar, valueAsString, originalValue)
    }
}

private fun <T> TableSpec<T>.appendClippedOrAligned(
    sb: StringBuilder,
    cellValue: Any?,
    columnWidth: Int,
    align: Align,
    paddingChar: String,
    onNull: () -> Any?,
) {
    val valueAsString = cellValue?.toString() ?: onNull().toString()

    if (ignoreAnsiEscapeSequences) {
        appendClippedOrAlignedIgnoringAnsiEsq(sb, cellValue, valueAsString, columnWidth, align, paddingChar)
    } else {
        appendClippedOrAlignedHandlingAnsiEsq(sb, cellValue, valueAsString, columnWidth, align, paddingChar)
    }
}

internal fun <T> TableSpec<T>.appendTopLineTo(
    sb: StringBuilder,
) = appendSeparatorRowTo(
    sb = sb,
    leftEnd = lineCharset.topLeftCorner,
    padding = lineCharset.horizontalOutline,
    columnSeparator = lineCharset.topEnd,
    rightEnd = lineCharset.topRightCorner,
    fillWith = lineCharset.horizontalOutline,
)

internal fun <T> TableSpec<T>.appendHeaderRowTo(
    sb: StringBuilder,
) = appendValueRowTo(sb) { _, column ->
    appendClippedOrAligned(
        sb,
        headerFormatter(column.header.toString()),
        column.width,
        column.alignHeader,
        column.padWith,
        column.onNull,
    )
}

internal fun <T> TableSpec<T>.appendFooterRowTo(
    sb: StringBuilder,
) = appendValueRowTo(sb) { _, column ->
    appendClippedOrAligned(
        sb,
        footerFormatter(column.footer.toString()),
        column.width,
        column.align,
        column.padWith,
        column.onNull,
    )
}

internal fun <T> TableSpec<T>.appendHeaderSeparatorTo(
    sb: StringBuilder,
) = appendSeparatorRowTo(
    sb = sb,
    leftEnd = lineCharset.leftEnd,
    padding = lineCharset.horizontalLine,
    columnSeparator = lineCharset.intersection,
    rightEnd = lineCharset.rightEnd,
    fillWith = lineCharset.horizontalLine,
)

internal fun <T> TableSpec<T>.appendRowSeparatorTo(
    sb: StringBuilder,
) = appendSeparatorRowTo(
    sb = sb,
    leftEnd = lineCharset.leftEnd,
    padding = lineCharset.horizontalLine,
    columnSeparator = lineCharset.intersection,
    rightEnd = lineCharset.rightEnd,
    fillWith = lineCharset.horizontalLine,
)

internal fun <T> TableSpec<T>.appendBodyRowTo(
    sb: StringBuilder,
    rowIdx: Int,
    rowValue: T
) = appendValueRowTo(sb) { _, column ->
    val cellValue = if (column.values.size <= rowIdx) {
        column.valueProvider(rowValue)
    } else {
        column.values[rowIdx]
    }

    appendClippedOrAligned(
        sb,
        cellValue,
        column.width,
        column.align,
        column.padWith,
        column.onNull,
    )
}

internal fun <T> TableSpec<T>.appendBottomLineTo(
    sb: StringBuilder,
) = appendSeparatorRowTo(
    sb = sb,
    leftEnd = lineCharset.bottomLeftCorner,
    padding = lineCharset.horizontalOutline,
    columnSeparator = lineCharset.bottomEnd,
    rightEnd = lineCharset.bottomRightCorner,
    fillWith = lineCharset.horizontalOutline,
)
