package dev.coderoutine.tabulate

internal interface Row

/**
 * A set of characters that are used for rendering the lines of tables.
 *
 * This interface contains some ready to use [LineCharset] objects.
 */
interface LineCharset {
    val horizontalLine: String
    val horizontalOutline: String get() = horizontalLine
    val verticalLine: String
    val verticalOutline: String get() = verticalLine
    val intersection: String
    val leftEnd: String
    val rightEnd: String
    val topEnd: String
    val bottomEnd: String
    val topLeftCorner: String
    val topRightCorner: String
    val bottomLeftCorner: String
    val bottomRightCorner: String

    /**
     * Creates a copy of this [LineCharset] overwriting
     * the values specified as parameters.
     */
    fun copyWith(
        horizontalLine: String = this.horizontalLine,
        horizontalOutline: String = this.horizontalOutline,
        verticalLine: String = this.verticalLine,
        verticalOutline: String = this.verticalOutline,
        intersection: String = this.intersection,
        leftEnd: String = this.leftEnd,
        rightEnd: String = this.rightEnd,
        topEnd: String = this.topEnd,
        bottomEnd: String = this.bottomEnd,
        topLeftCorner: String = this.topLeftCorner,
        topRightCorner: String = this.topRightCorner,
        bottomLeftCorner: String = this.bottomLeftCorner,
        bottomRightCorner: String = this.bottomRightCorner,
    ) = of(
        horizontalLine = horizontalLine,
        horizontalOutline = horizontalOutline,
        verticalLine = verticalLine,
        verticalOutline = verticalOutline,
        intersection = intersection,
        leftEnd = leftEnd,
        rightEnd = rightEnd,
        topEnd = topEnd,
        bottomEnd = bottomEnd,
        topLeftCorner = topLeftCorner,
        topRightCorner = topRightCorner,
        bottomLeftCorner = bottomLeftCorner,
        bottomRightCorner = bottomRightCorner,
    )

    /**
     * Creates a copy of this [LineCharset] with the given [formatter]
     * applied to all fields. This is useful if you want to apply
     * formatting to your lines using ANSI escape sequences.
     *
     * If you only want to format specific characters of the [LineCharset],
     * use [copyWith] instead and replace those characters with the formatted version.
     */
    fun formattedWith(formatter: (String) -> String) = copyWith(
        horizontalLine = formatter(horizontalLine),
        horizontalOutline = formatter(horizontalOutline),
        verticalLine = formatter(verticalLine),
        verticalOutline = formatter(verticalOutline),
        intersection = formatter(intersection),
        leftEnd = formatter(leftEnd),
        rightEnd = formatter(rightEnd),
        topEnd = formatter(topEnd),
        bottomEnd = formatter(bottomEnd),
        topLeftCorner = formatter(topLeftCorner),
        topRightCorner = formatter(topRightCorner),
        bottomLeftCorner = formatter(bottomLeftCorner),
        bottomRightCorner = formatter(bottomRightCorner),
    )

    companion object {
        /**
         * Create a free style instance of [LineCharset].
         */
        fun of(
            horizontalLine: String,
            horizontalOutline: String,
            verticalLine: String,
            verticalOutline: String,
            intersection: String,
            leftEnd: String,
            rightEnd: String,
            topEnd: String,
            bottomEnd: String,
            topLeftCorner: String,
            topRightCorner: String,
            bottomLeftCorner: String,
            bottomRightCorner: String,
        ): LineCharset = LineCharsetData(
            horizontalLine = horizontalLine,
            horizontalOutline = horizontalOutline,
            verticalLine = verticalLine,
            verticalOutline = verticalOutline,
            intersection = intersection,
            leftEnd = leftEnd,
            rightEnd = rightEnd,
            topEnd = topEnd,
            bottomEnd = bottomEnd,
            topLeftCorner = topLeftCorner,
            topRightCorner = topRightCorner,
            bottomLeftCorner = bottomLeftCorner,
            bottomRightCorner = bottomRightCorner,
        )
    }

    /**
     * Simple lines using only "|" and "-".
     */
    data object Simple : LineCharset {
        override val horizontalLine = "-"
        override val verticalLine = "|"
        override val intersection = verticalLine
        override val leftEnd get() = verticalLine
        override val rightEnd get() = verticalLine
        override val topEnd get() = verticalLine
        override val bottomEnd get() = verticalLine
        override val topLeftCorner get() = verticalLine
        override val topRightCorner get() = verticalLine
        override val bottomLeftCorner get() = verticalLine
        override val bottomRightCorner get() = verticalLine
    }

    /**
     * Thin lines from the box drawing character set.
     */
    data object Thin : LineCharset {
        override val horizontalLine = "─"
        override val verticalLine = "│"
        override val intersection = "┼"
        override val leftEnd = "├"
        override val rightEnd = "┤"
        override val topEnd = "┬"
        override val bottomEnd = "┴"
        override val topLeftCorner = "┌"
        override val topRightCorner = "┐"
        override val bottomLeftCorner = "└"
        override val bottomRightCorner = "┘"
    }

    /**
     * Thick lines from the box drawing character set.
     */
    data object Thick : LineCharset {
        override val horizontalLine = "━"
        override val verticalLine = "┃"
        override val intersection = "╋"
        override val leftEnd = "┣"
        override val rightEnd = "┫"
        override val topEnd = "┳"
        override val bottomEnd = "┻"
        override val topLeftCorner = "┏"
        override val topRightCorner = "┓"
        override val bottomLeftCorner = "┗"
        override val bottomRightCorner = "┛"
    }

    /**
     * Thick outer and thin inner lines from the box drawing character set.
     */
    data object ThickOutline : LineCharset by Thick {
        override val horizontalLine get() = Thin.horizontalOutline
        override val verticalLine get() = Thin.verticalOutline
        override val intersection get() = Thin.intersection
        override val leftEnd = "┠"
        override val rightEnd = "┨"
        override val topEnd = "┯"
        override val bottomEnd = "┷"
    }

    /**
     * Thick lines from the box drawing character set with rounded corners.
     */
    data object ThinWithRoundCorners : LineCharset by Thin {
        override val topLeftCorner = "╭"
        override val topRightCorner = "╮"
        override val bottomLeftCorner = "╰"
        override val bottomRightCorner = "╯"
    }

    /**
     * Doubled lines from the box drawing character set.
     */
    data object Doubled : LineCharset {
        override val horizontalLine = "═"
        override val verticalLine = "║"
        override val intersection = "╬"
        override val leftEnd = "╠"
        override val rightEnd = "╣"
        override val topEnd = "╦"
        override val bottomEnd = "╩"
        override val topLeftCorner = "╔"
        override val topRightCorner = "╗"
        override val bottomLeftCorner = "╚"
        override val bottomRightCorner = "╝"
    }

    /**
     * Doubled outer and thin inner lines from the box drawing character set.
     */
    data object DoubledOutline : LineCharset by Doubled {
        override val horizontalLine get() = Thin.horizontalOutline
        override val verticalLine get() = Thin.verticalOutline
        override val intersection get() = Thin.intersection
        override val leftEnd = "╟"
        override val rightEnd = "╢"
        override val topEnd = "╤"
        override val bottomEnd = "╧"
    }
}

private data class LineCharsetData(
    override val horizontalLine: String,
    override val horizontalOutline: String,
    override val verticalLine: String,
    override val verticalOutline: String,
    override val intersection: String,
    override val leftEnd: String,
    override val rightEnd: String,
    override val topEnd: String,
    override val bottomEnd: String,
    override val topLeftCorner: String,
    override val topRightCorner: String,
    override val bottomLeftCorner: String,
    override val bottomRightCorner: String
) : LineCharset