package dev.coderoutine.tabulate

import dev.coderoutine.tabulate.Align.DYNAMIC

private typealias RowPredicate = (rowIdx: Int, isLast: Boolean) -> Boolean

class TableSpec<T> {
    /**
     * How to align the values in the cells.
     */
    var align: Align = DYNAMIC

    /**
     * Whether to print the headers.
     */
    var withHeader: Boolean = true

    /**
     * How to align the headers.
     */
    var alignHeader: Align = align

    /**
     * How to align the footers.
     */
    var alignFooter: Align = align

    /**
     * Whether to add a row at the end with special values.
     * The value to can be defined per column using the footer argument.
     */
    var withFooter: Boolean = false

    /**
     * The character that should be used to fill empty positions when aligning values in cells.
     */
    var padWith: String = " "

    /**
     * How many positions should be padded between cell values and the column separators.
     * This value only applies horizontally.
     */
    var paddingWidth: Int = 1

    /**
     * Global value provider as replacement for `null` values.
     * By default, `null` is used as is.
     */
    var onNull: () -> Any? = { null }

    var ignoreAnsiEscapeSequences: Boolean = false

    internal var lineCharset: LineCharset = LineCharset.Simple
    internal var includeColumnSeparator: Boolean = true
    internal var includeHeaderSeparator: Boolean = false
    internal var includeLeftRightOutline: Boolean = false
    internal var includeTopBottomOutline: Boolean = false
    internal var rowSeparatorPredicate: RowPredicate = { _, _ -> false }

    internal val columns: MutableList<ColumnSpec<T>> = mutableListOf()
    internal var headerFormatter: (String) -> Any? = { it }
    internal var footerFormatter: (String) -> Any? = { it }

    /**
     * Define which lines to print and which characters to use for it.
     */
    fun lines(
        charset: LineCharset = this.lineCharset,
        betweenColumns: Boolean = this.includeColumnSeparator,
        afterHeader: Boolean = this.includeHeaderSeparator,
        leftRightOutline: Boolean = this.includeLeftRightOutline,
        topBottomOutline: Boolean = this.includeTopBottomOutline,
        afterRow: RowPredicate = this.rowSeparatorPredicate,
    ) {
        this.lineCharset = charset
        this.includeColumnSeparator = betweenColumns
        this.includeHeaderSeparator = afterHeader
        this.includeLeftRightOutline = leftRightOutline
        this.includeTopBottomOutline = topBottomOutline
        this.rowSeparatorPredicate = afterRow
    }

    /**
     * Add a column to the table with the given properties.
     */
    fun column(
        /**
         * The header of this column.
         */
        header: Any = "",
        /**
         * The footer value of this column. Default is `""`.
         */
        footer: Any = "",
        /**
         * A custom alignment for the cells that only applies to this column.
         * Defaults to [TableSpec.align].
         */
        align: Align = this.align,
        /**
         * A custom alignment for the header that only applies to this column.
         * Defaults to [TableSpec.alignHeader].
         */
        alignHeader: Align = this.alignHeader,
        /**
         * A custom alignment for the header that only applies to this column.
         * Defaults to [TableSpec.alignFooter].
         */
        alignFooter: Align = this.alignFooter,
        /**
         * The character that should be used to fill empty positions when aligning values in cells of this column.
         * Defaults to [TableSpec.paddingWidth].
         */
        padWith: String = this.padWith,
        /**
         * A fixed width for this column or `-1` if the column should be measured.
         * Is `-1` by default.
         */
        width: Int = -1,
        /**
         * Minimum width of this column.
         */
        minWidth: Int = 0,
        /**
         * Maximum width of this column.
         */
        maxWidth: Int = Int.MAX_VALUE,
        /**
         * Column specific value provider as replacement for `null` values.
         * By default, [TableSpec.onNull] is used.
         */
        onNull: () -> Any? = this.onNull,
        /**
         * A function that extracts the value of this column from each instance of [T].
         * This function will be called once for each [T] in the given data.
         */
        valueProvider: (T) -> Any?,
    ) {
        columns.add(
            ColumnSpec(
                header = header,
                align = align,
                alignHeader = alignHeader,
                alignFooter = alignFooter,
                padWith = padWith,
                width = width,
                minWidth = minWidth,
                maxWidth = maxWidth,
                onNull = onNull,
                footer = footer,
                ignoreAnsiEscapeSequences = ignoreAnsiEscapeSequences,
                valueProvider = valueProvider,
            )
        )
    }

    fun formatHeaders(formatter: (String) -> Any?) {
        headerFormatter = formatter
    }

    fun formatFooters(formatter: (String) -> Any?) {
        footerFormatter = formatter
    }

    internal fun measureColumns(data: Data<T>) {
        mainLoop@ for (column in columns) {
            if (column.width >= 0) {
                continue
            }

            if (column.updateWidthWithinConstraints(column.header)) {
                // Skip iterating rest of data because maxWidth is already reached.
                continue
            }

            if (column.updateWidthWithinConstraints(column.footer)) {
                // Skip iterating rest of data because maxWidth is already reached.
                continue
            }

            for (record in data.measurementIterator()) {
                val value = column.valueProvider(record)
                column.values.add(value)

                if (column.updateWidthWithinConstraints(value)) {
                    // Skip iterating rest of data because maxWidth is already reached.
                    continue@mainLoop
                }
            }
        }
    }

}