package dev.shortloop.agent;

import dev.shortloop.agent.buffer.ApiBufferKey;
import dev.shortloop.agent.commons.ObservedApi;
import dev.shortloop.agent.config.ConfigManager;
import dev.shortloop.agent.config.ConfigUpdateListener;
import dev.shortloop.common.models.constant.ShortloopCommonConstant;
import dev.shortloop.common.models.data.AgentConfig;
import dev.shortloop.common.models.data.BlackListRule;
import lombok.*;
import lombok.experimental.FieldDefaults;
import org.springframework.core.annotation.Order;
import org.springframework.util.CollectionUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

@Order
public class ShortloopSpringFilter extends OncePerRequestFilter implements ConfigUpdateListener {
    private AgentConfig agentConfig;
    private ConfigManager configManager;

    private ShortloopApiProcessor apiProcessor;

    private String userApplicationName;

    private SDKLogger logger;

    public ShortloopSpringFilter(ConfigManager configManager, ShortloopApiProcessor apiProcessor, String userApplicationName, SDKLogger logger) {
        this.configManager = configManager;
        this.apiProcessor = apiProcessor;
        this.userApplicationName = userApplicationName;
        this.logger = logger;
    }

    public boolean init() {
        configManager.subscribeToUpdates(this);
        return true;
    }

    @Getter
    @Setter
    @RequiredArgsConstructor
    @FieldDefaults(level = AccessLevel.PRIVATE)
    public static class RequestResponseContext {
        final HttpServletRequest servletRequest;
        final HttpServletResponse servletResponse;
        final String applicationName;
        ContentCachingRequestWrapper cachedRequest;
        ContentCachingResponseWrapper cachedResponse;
        ObservedApi observedApi;
        AgentConfig.ApiConfig apiConfig;
        AgentConfig agentConfig;
        ApiBufferKey apiBufferKey;
        Boolean payloadCaptureAttempted;
        Boolean requestPayloadCaptureAttempted;
        Boolean responsePayloadCaptureAttempted;
        Long latency;
    }

    @Override
    protected void doFilterInternal(
            final HttpServletRequest servletRequest,
            final HttpServletResponse servletResponse,
            final FilterChain filterChain)
            throws ServletException, IOException {
        AgentConfig agentConfigLocal = this.agentConfig;
        if (null == agentConfigLocal) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        if (null == agentConfigLocal.getCaptureApiSample() ||
                !agentConfigLocal.getCaptureApiSample()) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        ObservedApi observedApi = getObservedApiFromRequest(servletRequest);
        if (isBlackListedApi(observedApi, agentConfigLocal)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        RequestResponseContext context = new RequestResponseContext(servletRequest, servletResponse,
                userApplicationName);
        context.setObservedApi(observedApi);
        AgentConfig.ApiConfig apiConfig = getApiConfig(observedApi, agentConfigLocal);
        context.setAgentConfig(agentConfigLocal);

        if (null != apiConfig) {
            context.setApiConfig(apiConfig);
            context.setApiBufferKey(ApiBufferKey.getApiBufferKeyFrom(context.getApiConfig()));
            apiProcessor.processRegisteredApi(context, filterChain);
        } else {
            context.setApiBufferKey(ApiBufferKey.getApiBufferKeyFrom(context.getObservedApi()));
            apiProcessor.processDiscoveredApi(context, filterChain);
        }
    }

    private boolean isBlackListedApi(ObservedApi observedApi, AgentConfig agentConfig) {
        try {
            if (null == agentConfig || null == agentConfig.getBlackListRules()) {
                return false;
            }
            for (BlackListRule blackListRule : agentConfig.getBlackListRules()) {
                if (blackListRule.matchesUri(observedApi.getUri(), observedApi.getMethod())) {
                    return true;
                }
            }
        } catch (Exception e) {
            logger.error("Error ShortloopSpringFilter::isBlackListedApi", e);
        }
        return false;
    }

    private AgentConfig.ApiConfig getApiConfig(ObservedApi observedApi, AgentConfig agentConfig) {
        if (CollectionUtils.isEmpty(agentConfig.getRegisteredApiConfigs())) {
            return null;
        }
        List<AgentConfig.ApiConfig> registeredApis = agentConfig.getRegisteredApiConfigs();
        for(AgentConfig.ApiConfig apiConfig: registeredApis) {
            if (observedApi.matches(apiConfig)) {
                return apiConfig;
            }
        }
        return null;
    }

    private ObservedApi getObservedApiFromRequest(HttpServletRequest servletRequest) {
        return new ObservedApi(servletRequest.getRequestURI(), ShortloopCommonConstant.HttpRequestMethod.valueOf(servletRequest.getMethod()));
    }

    @Override
    public void onSuccessfulConfigUpdate(AgentConfig agentConfig) {
        this.agentConfig = agentConfig;
    }

    @Override
    public void onErroneousConfigUpdate() {
        this.agentConfig = AgentConfig.noOpAgentConfig;
    }

}
