package dev.shortloop.agent.buffer;

import dev.shortloop.agent.ShortloopHTTPConnection;
import dev.shortloop.agent.SDKLogger;
import dev.shortloop.agent.config.ConfigManager;
import dev.shortloop.common.models.data.APISample;
import dev.shortloop.common.models.data.AgentConfig;
import lombok.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Semaphore;
public class DiscoveredApiBufferManager extends AbstractBufferManager {
    private final ShortloopHTTPConnection shortloopHTTPConnection;
    private final String ctUrl;
    private final SDKLogger logger;
    public DiscoveredApiBufferManager(ConfigManager configManager, ShortloopHTTPConnection shortloopHTTPConnection, String ctUrl, SDKLogger logger) {
        super(configManager, logger);
        this.shortloopHTTPConnection = shortloopHTTPConnection;
        this.ctUrl = ctUrl;
        this.logger = logger;
    }

    BufferManagerWorker<ApiBufferKey> createWorker(AgentConfig newConfig) {
        return new DiscoveredApiBufferManagerWorker(newConfig, shortloopHTTPConnection, logger);
    }

    public class DiscoveredApiBufferManagerWorker extends BufferManagerWorker<ApiBufferKey> {

        private final ShortloopHTTPConnection shortloopHTTPConnection;
        private final Semaphore semaphore;

        public DiscoveredApiBufferManagerWorker(@NonNull AgentConfig config, ShortloopHTTPConnection shortloopHTTPConnection, SDKLogger logger) {
            super(config, ctUrl, logger);
            semaphore = new Semaphore(config.getDiscoveryBufferSize());
            this.shortloopHTTPConnection = shortloopHTTPConnection;
        }

        @Override
        public boolean init() {
            return true;
        }

        public boolean offer(ApiBufferKey apiBufferKey, APISample apiSample) {
            bufferMap.computeIfAbsent(apiBufferKey, (key) -> new SimpleBuffer(getOperatingConfig().getDiscoveryBufferSizePerApi()));
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (buffer != null) {
                return buffer.offer(apiSample);
            } else {
                logger.error("Buffer found null for " + apiBufferKey.getUri());
            }
            return false;
        }

        public boolean canOffer(ApiBufferKey apiBufferKey) {
            if (semaphore.tryAcquire()) {
                boolean canOffer = false;
                if (getOperatingConfig().getCaptureApiSample()) {
                    Buffer buffer = bufferMap.get(apiBufferKey);
                    if (null != buffer) {
                        canOffer = buffer.canOffer();
                    } else {
                        canOffer = bufferMap.size() < getOperatingConfig().getDiscoveryBufferSize();
                    }
                }
                semaphore.release();
                return canOffer;
            }
            return false;
        }
        void syncForKey(ApiBufferKey key) {
            try {
                Buffer buffer = bufferMap.get(key);
                if (null == buffer) {
                    logger.error("Buffer is null for " + key.getUri());
                    return;
                }
                int iterations = buffer.getContentCount();
                if (iterations == 0) {
                    bufferMap.remove(key);
                    return;
                }
                List<APISample> contents = new ArrayList<>();
                while (iterations -- > 0) {
                    APISample apiSample = buffer.poll();
                    if (null == apiSample) {
                        bufferMap.remove(key);
                        break;
                    }
                    contents.add(apiSample);
                }
                if (contents.size() == 0) {
                    return;
                }
                this.shortloopHTTPConnection.sendSamples(contents);
            } catch (Exception exception) {
                logger.error("Error inside syncForKey for key " + key.getUri(), exception);
            }
        }
    }
}
