/*
 * Decompiled with CFR 0.152.
 */
package dev.voidframework.scheduler;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Singleton;
import com.typesafe.config.Config;
import dev.voidframework.core.bindable.BindClass;
import dev.voidframework.core.lifecycle.LifeCycleStart;
import dev.voidframework.core.lifecycle.LifeCycleStop;
import dev.voidframework.scheduler.cron.CronExpression;
import dev.voidframework.scheduler.exception.SchedulerException;
import dev.voidframework.scheduler.module.ScheduledHandlers;
import java.time.ZoneId;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@BindClass
@Singleton
public final class SchedulerManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SchedulerManager.class);
    private final Injector injector;
    private final ScheduledExecutorService scheduledExecutorService;

    @Inject
    public SchedulerManager(Config configuration, Injector injector) {
        this.injector = injector;
        this.scheduledExecutorService = Executors.newScheduledThreadPool(configuration.getInt("voidframework.scheduler.threadPoolSize"), new SchedulerThreadFactory());
    }

    @LifeCycleStart(priority=600)
    public void startScheduler() {
        ScheduledHandlers scheduledHandlers = (ScheduledHandlers)this.injector.getInstance(ScheduledHandlers.class);
        for (ScheduledHandlers.ScheduledHandler scheduledHandler : scheduledHandlers) {
            if (StringUtils.isNotBlank((CharSequence)scheduledHandler.scheduledAnnotation().cron())) {
                this.registerCron(scheduledHandler);
                continue;
            }
            this.registerDelay(scheduledHandler);
        }
    }

    @LifeCycleStop(priority=1)
    public void stopScheduler() {
        try {
            this.scheduledExecutorService.shutdownNow();
        }
        catch (Exception exception) {
            LOGGER.error("An error occur during the scheduler termination", (Throwable)exception);
        }
    }

    private void registerCron(final ScheduledHandlers.ScheduledHandler scheduledHandler) {
        final CronExpression cronExpression = new CronExpression(scheduledHandler.scheduledAnnotation().cron());
        final ZoneId zoneId = ZoneId.of(scheduledHandler.scheduledAnnotation().cronZone());
        final Object classInstance = this.injector.getInstance(scheduledHandler.classType());
        LOGGER.info("Method {}::{} is scheduled using CRON expression \"{}\" (TimeZone: {})", new Object[]{scheduledHandler.classType().getName(), scheduledHandler.method().getName(), scheduledHandler.scheduledAnnotation().cron(), zoneId});
        Callable<Void> callable = new Callable<Void>(){

            @Override
            public Void call() {
                SchedulerManager.this.scheduledExecutorService.schedule(this, cronExpression.getNextDelayMilliseconds(zoneId), TimeUnit.MILLISECONDS);
                try {
                    scheduledHandler.method().invoke(classInstance, new Object[0]);
                }
                catch (Exception exception) {
                    LOGGER.error("An error occurred during the execution of scheduled method {}::{}", new Object[]{scheduledHandler.getClass().getName(), scheduledHandler.method().getName(), exception});
                }
                return null;
            }
        };
        this.scheduledExecutorService.schedule(callable, cronExpression.getNextDelayMilliseconds(zoneId), TimeUnit.MILLISECONDS);
    }

    private void registerDelay(final ScheduledHandlers.ScheduledHandler scheduledHandler) {
        if (scheduledHandler.scheduledAnnotation().fixedDelay() < 0) {
            throw new SchedulerException.InvalidFixedDelay(scheduledHandler.scheduledAnnotation().fixedRate());
        }
        if (scheduledHandler.scheduledAnnotation().fixedRate() < 0) {
            throw new SchedulerException.InvalidFixedRate(scheduledHandler.scheduledAnnotation().fixedRate());
        }
        if (scheduledHandler.scheduledAnnotation().initialDelay() < 0) {
            throw new SchedulerException.InvalidInitialDelay(scheduledHandler.scheduledAnnotation().initialDelay());
        }
        if (scheduledHandler.scheduledAnnotation().fixedDelay() >= 1 && scheduledHandler.scheduledAnnotation().fixedRate() >= 1) {
            throw new SchedulerException.FixedDelayAndRateAreExclusive();
        }
        final Object classInstance = this.injector.getInstance(scheduledHandler.classType());
        int initialDelay = scheduledHandler.scheduledAnnotation().initialDelay() > 0 ? scheduledHandler.scheduledAnnotation().initialDelay() : scheduledHandler.scheduledAnnotation().fixedRate();
        LOGGER.info("Method {}::{} is scheduled to be run every {} seconds{}", new Object[]{scheduledHandler.classType().getName(), scheduledHandler.method().getName(), scheduledHandler.scheduledAnnotation().fixedDelay() >= 1 ? scheduledHandler.scheduledAnnotation().fixedDelay() : scheduledHandler.scheduledAnnotation().fixedRate(), scheduledHandler.scheduledAnnotation().initialDelay() > 0 ? " after an initial delay of %d seconds".formatted(scheduledHandler.scheduledAnnotation().initialDelay()) : ""});
        Callable<Void> callable = scheduledHandler.scheduledAnnotation().fixedDelay() >= 1 ? new Callable<Void>(){

            @Override
            public Void call() {
                try {
                    scheduledHandler.method().invoke(classInstance, new Object[0]);
                }
                catch (Exception exception) {
                    LOGGER.error("An error occurred during the execution of scheduled method {}::{}", new Object[]{scheduledHandler.getClass().getName(), scheduledHandler.method().getName(), exception});
                }
                SchedulerManager.this.scheduledExecutorService.schedule(this, (long)scheduledHandler.scheduledAnnotation().fixedDelay(), TimeUnit.MILLISECONDS);
                return null;
            }
        } : new Callable<Void>(){

            @Override
            public Void call() {
                SchedulerManager.this.scheduledExecutorService.schedule(this, (long)scheduledHandler.scheduledAnnotation().fixedRate(), TimeUnit.MILLISECONDS);
                try {
                    scheduledHandler.method().invoke(classInstance, new Object[0]);
                }
                catch (Exception exception) {
                    LOGGER.error("An error occurred during the execution of scheduled method {}::{}", new Object[]{scheduledHandler.getClass().getName(), scheduledHandler.method().getName(), exception});
                }
                return null;
            }
        };
        this.scheduledExecutorService.schedule(callable, (long)initialDelay, TimeUnit.MILLISECONDS);
    }

    private static final class SchedulerThreadFactory
    implements ThreadFactory {
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;

        public SchedulerThreadFactory() {
            this.group = Thread.currentThread().getThreadGroup();
            this.namePrefix = "scheduler-";
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = new Thread(this.group, runnable, this.namePrefix + this.threadNumber.getAndIncrement(), 0L);
            if (thread.isDaemon()) {
                thread.setDaemon(false);
            }
            if (thread.getPriority() != 5) {
                thread.setPriority(5);
            }
            return thread;
        }
    }
}

