/*
 * Decompiled with CFR 0.152.
 */
package dev.voidframework.web.http.filter.csrf;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import dev.voidframework.core.lang.TypedMap;
import dev.voidframework.core.utils.HexUtils;
import dev.voidframework.web.exception.HttpException;
import dev.voidframework.web.http.Context;
import dev.voidframework.web.http.Cookie;
import dev.voidframework.web.http.FormData;
import dev.voidframework.web.http.FormItem;
import dev.voidframework.web.http.HttpMethod;
import dev.voidframework.web.http.Result;
import dev.voidframework.web.http.filter.Filter;
import dev.voidframework.web.http.filter.FilterChain;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

@Singleton
public class CSRFFilter
implements Filter {
    public static final TypedMap.Key<String> CSRF_TOKEN_KEY = TypedMap.Key.of((String)"CSRF_TOKEN", String.class);
    public static final TypedMap.Key<Boolean> BYPASS_CSRF_VERIFICATION = TypedMap.Key.of((String)"BYPASS_CSRF_VERIFICATION", Boolean.class);
    private static final String H_MAC_ALGORITHM = "HmacSHA256";
    private final String csrfTokenName;
    private final String cookieName;
    private final boolean cookieSecure;
    private final boolean cookieHttpOnly;
    private final String signatureKey;
    private final long timeToLive;

    @Inject
    public CSRFFilter(Config configuration) {
        this.csrfTokenName = configuration.getString("voidframework.web.csrf.tokenName");
        this.cookieName = configuration.getString("voidframework.web.csrf.cookieName");
        this.cookieSecure = configuration.getBoolean("voidframework.web.csrf.cookieSecure");
        this.cookieHttpOnly = configuration.getBoolean("voidframework.web.csrf.cookieHttpOnly");
        this.signatureKey = configuration.getString("voidframework.web.csrf.signatureKey");
        this.timeToLive = configuration.getDuration("voidframework.web.csrf.timeToLive", TimeUnit.MILLISECONDS);
        if (StringUtils.isBlank((CharSequence)this.signatureKey)) {
            throw new ConfigException.BadValue("voidframework.web.csrf.signatureKey", "Please configure the CSRF signature Key");
        }
    }

    @Override
    public Result apply(Context context, FilterChain filterChain) {
        if (context.getAttributes().get(BYPASS_CSRF_VERIFICATION) == Boolean.TRUE) {
            return filterChain.applyNext(context);
        }
        if (context.getRequest().getHttpMethod() == HttpMethod.POST) {
            Pair<CSRFToken, String> currentAndNewCSRFTokenPair = this.extractAndRegenerateCSRFToken(context);
            CSRFToken providedCSRFToken = this.extractProvidedCSRFToken(context);
            this.checkCSRFToken((CSRFToken)currentAndNewCSRFTokenPair.getLeft(), providedCSRFToken);
            Result result = filterChain.applyNext(context);
            if (result.getHttpCode() / 100 != 3) {
                return result.withCookie(Cookie.of(this.cookieName, (String)currentAndNewCSRFTokenPair.getRight(), this.cookieHttpOnly, this.cookieSecure, null));
            }
            return result.withoutCookie(this.cookieName);
        }
        if (context.getRequest().getHttpMethod() == HttpMethod.GET && context.getRequest().acceptContentType("text/html")) {
            Pair<CSRFToken, String> tokens = this.extractAndRegenerateCSRFToken(context);
            return filterChain.applyNext(context).withCookie(Cookie.of(this.cookieName, (String)tokens.getRight(), this.cookieHttpOnly, this.cookieSecure, null));
        }
        return filterChain.applyNext(context);
    }

    private Pair<CSRFToken, String> extractAndRegenerateCSRFToken(Context context) {
        CSRFToken currentCSRFToken = this.extractCurrentCSRFToken(context);
        String newCSRFTokenAsString = this.createNewCSRFTokenAsString(currentCSRFToken);
        context.getAttributes().put(CSRF_TOKEN_KEY, (Object)newCSRFTokenAsString);
        return Pair.of((Object)currentCSRFToken, (Object)newCSRFTokenAsString);
    }

    private CSRFToken extractCurrentCSRFToken(Context context) {
        Cookie csrfTokenCookie = context.getRequest().getCookie(this.cookieName);
        if (csrfTokenCookie == null) {
            return null;
        }
        String[] csrfTokenPartArray = csrfTokenCookie.value().split("-");
        if (csrfTokenPartArray.length != 3) {
            return null;
        }
        return new CSRFToken(csrfTokenPartArray[0], Long.parseLong(csrfTokenPartArray[1]), csrfTokenPartArray[2]);
    }

    private CSRFToken extractProvidedCSRFToken(Context context) {
        List csrfFormItem;
        String csrfTokenProvided = context.getRequest().getQueryStringParameter(this.csrfTokenName);
        if (StringUtils.isNotBlank((CharSequence)csrfTokenProvided)) {
            String[] csrfTokenPartArray = csrfTokenProvided.split("-");
            if (csrfTokenPartArray.length != 3) {
                return null;
            }
            return new CSRFToken(csrfTokenPartArray[0], Long.parseLong(csrfTokenPartArray[1]), csrfTokenPartArray[2]);
        }
        FormData formData = context.getRequest().getBodyContent().asFormData();
        if (formData != null && !(csrfFormItem = formData.getOrDefault(this.csrfTokenName, List.of())).isEmpty() && StringUtils.isNotBlank((CharSequence)(csrfTokenProvided = ((FormItem)csrfFormItem.get(0)).value()))) {
            String[] csrfTokenPartArray = csrfTokenProvided.split("-");
            if (csrfTokenPartArray.length != 3) {
                return null;
            }
            return new CSRFToken(csrfTokenPartArray[0], Long.parseLong(csrfTokenPartArray[1]), csrfTokenPartArray[2]);
        }
        csrfTokenProvided = context.getRequest().getHeader("X-CSRF-TOKEN");
        if (StringUtils.isNotBlank((CharSequence)csrfTokenProvided)) {
            String[] csrfTokenPartArray = csrfTokenProvided.split("-");
            if (csrfTokenPartArray.length != 3) {
                return null;
            }
            return new CSRFToken(csrfTokenPartArray[0], Long.parseLong(csrfTokenPartArray[1]), csrfTokenPartArray[2]);
        }
        return null;
    }

    private String createNewCSRFTokenAsString(CSRFToken existingCSRFToken) {
        long currentTimeMillis = System.currentTimeMillis();
        String value = existingCSRFToken != null ? existingCSRFToken.value : UUID.randomUUID().toString().replace("-", "");
        return new CSRFToken(value, currentTimeMillis, this.generateSignature(value + currentTimeMillis)).toString();
    }

    private String generateSignature(String value) {
        try {
            Mac mac = Mac.getInstance(H_MAC_ALGORITHM);
            mac.init(new SecretKeySpec(this.signatureKey.getBytes(StandardCharsets.UTF_8), H_MAC_ALGORITHM));
            return HexUtils.toHex((byte[])mac.doFinal(value.getBytes(StandardCharsets.UTF_8)));
        }
        catch (InvalidKeyException | NoSuchAlgorithmException exception) {
            throw new HttpException.InternalServerError(exception);
        }
    }

    private void checkCSRFToken(CSRFToken csrfTokenExpected, CSRFToken csrfTokenProvided) {
        String providedValue;
        if (csrfTokenExpected == null || csrfTokenProvided == null) {
            throw new HttpException.BadRequest("CSRF token not found");
        }
        if (csrfTokenProvided.nonce + this.timeToLive < System.currentTimeMillis()) {
            throw new HttpException.BadRequest("CSRF token is expired");
        }
        String expectedValue = Objects.equals(csrfTokenExpected.signature, this.generateSignature(csrfTokenExpected.value + csrfTokenExpected.nonce)) ? csrfTokenExpected.value : null;
        String string = providedValue = Objects.equals(csrfTokenProvided.signature, this.generateSignature(csrfTokenProvided.value + csrfTokenProvided.nonce)) ? csrfTokenProvided.value : null;
        if (expectedValue == null || !Objects.equals(expectedValue, providedValue)) {
            throw new HttpException.BadRequest("CSRF token is invalid");
        }
    }

    private record CSRFToken(String value, long nonce, String signature) {
        @Override
        public String toString() {
            return this.value + "-" + this.nonce + "-" + this.signature;
        }
    }
}

