package fi.evolver.script;

import java.io.IOException;
import java.util.*;

import com.googlecode.lanterna.TerminalSize;
import com.googlecode.lanterna.gui2.*;
import com.googlecode.lanterna.screen.Screen;
import com.googlecode.lanterna.screen.TerminalScreen;
import com.googlecode.lanterna.screen.VirtualScreen;
import com.googlecode.lanterna.terminal.DefaultTerminalFactory;
import com.googlecode.lanterna.terminal.Terminal;

// TODO port from Zenity to Yad if graphical dialog is needed
//  Yad is much more powerful and flexible than Zenity supporting e.g. multiline text fields in forms

public class Dialog {
	private static final String SEPARATOR = UUID.randomUUID().toString();

	private static final LayoutData LANTERNA_LAYOUT_SPAN_END = GridLayout.createLayoutData(
			GridLayout.Alignment.END,
			GridLayout.Alignment.CENTER,
			true,
			false,
			2,
			1
	);

	private static final LayoutData LANTERNA_LAYOUT_SPAN_BEGINNING = GridLayout.createLayoutData(
			GridLayout.Alignment.BEGINNING,
			GridLayout.Alignment.CENTER,
			true,
			false,
			2,
			1
	);

	// Can be used to set the default UI mode for all dialogs
	public static UiMode DEFAULT_UI_MODE = UiMode.LANTERNA;

	private final List<DialogEntry> entries = new ArrayList<>();
	private final String title;
	private UiMode uiMode;

	public Dialog(String title) {
		this.title = title;
		this.uiMode = DEFAULT_UI_MODE;
	}

	public Dialog add(DialogEntry entry) {
		this.entries.add(entry);
		return this;
	}

	public Dialog setUiMode(UiMode mode) {
		this.uiMode = mode;
		return this;
	}

	public SequencedMap<DialogEntry, String> show() {
		if (entries.isEmpty())
			return Collections.emptySortedMap();

		UiMode effectiveMode = determineEffectiveUiMode();
		return switch (effectiveMode) {
			case ZENITY -> showZenityDialog();
			case LANTERNA -> showLanternaDialog();
		};
	}

	private UiMode determineEffectiveUiMode() {
		if (uiMode == UiMode.LANTERNA) {
			return UiMode.LANTERNA;
		}

		try {
			Process process = new ProcessBuilder("which", "zenity").start();
			int exitCode = process.waitFor();
			boolean hasZenity = exitCode == 0;
			return hasZenity ? UiMode.ZENITY : UiMode.LANTERNA;
		} catch (IOException | InterruptedException e) {
			return UiMode.LANTERNA; // Fallback to terminal UI
		}
	}

	private SequencedMap<DialogEntry, String> showZenityDialog() {
		// TODO: Zenity forms have limited support for multiline input fields and no native checkbox/radio support

		List<String> command = new ArrayList<>();
		command.add("/usr/bin/zenity");
		command.add("--forms");
		command.add("--ok-label");
		command.add("Submit");
		command.add("--title");
		command.add(title);
		command.add("--text");
		command.add(title);
		command.add("--separator");
		command.add(SEPARATOR);

		for (DialogEntry entry : entries) {
			switch (entry) {
				case DialogParagraph paragraph -> {
					// Zenity forms don't directly support text paragraphs. Using label only.
					command.add("--add-entry");
					command.add(paragraph.label());
					command.add("--add-entry");
					command.add(""); // Add empty field to maintain indexing with results
				}
				case DialogTextField textField -> {
					boolean isMultiline = textField.rows() > 1;
					if (isMultiline) {
						System.err.println("Warning: Zenity has limited support for multiline fields. Consider using Lanterna UI mode instead.");
					}

					command.add(textField.isPassword() ? "--add-password" : "--add-entry");
					command.add(textField.label());
				}
				case DialogCheckBox checkBox -> {
					System.err.println("Warning: Zenity forms don't support native checkboxes. Using text field instead.");
					command.add("--add-entry");
					command.add(checkBox.label() + " (Enter 'true' or 'false')");
				}
				case DialogRadioSelect radioButton -> {
					System.err.println("Warning: Zenity forms don't support native radio buttons. Using text field instead.");
					StringBuilder optionsText = new StringBuilder(radioButton.label())
							.append(" (Enter one of: ");
					List<String> options = radioButton.options();
					for (int i = 0; i < options.size(); i++) {
						if (i > 0) optionsText.append(", ");
						optionsText.append("'").append(options.get(i)).append("'");
					}
					optionsText.append(")");

					command.add("--add-entry");
					command.add(optionsText.toString());
				}
			}
		}

		String output = Shell.user(command);
		String[] values = output.split(SEPARATOR);

		SequencedMap<DialogEntry, String> results = new LinkedHashMap<>();
		int valueIndex = 0;
		for (DialogEntry entry : entries) {
			if (valueIndex < values.length) {
				results.put(entry, values[valueIndex].replaceFirst("\n$", ""));
				valueIndex++;
			}
		}

		return results;
	}

	private SequencedMap<DialogEntry, String> showLanternaDialog() {
		try {
			DefaultTerminalFactory terminalFactory = new DefaultTerminalFactory();
			Terminal terminal = terminalFactory.createTerminal();
			Screen screen = new TerminalScreen(terminal);
			screen.doResizeIfNecessary();
			screen.startScreen();

			BasicWindow window = new BasicWindow(title);
			window.setHints(List.of(Window.Hint.MODAL));


			Panel contentPanel = new Panel();
			contentPanel.setLayoutManager(new GridLayout(2).setHorizontalSpacing(1).setVerticalSpacing(1));

			Map<DialogEntry, Component> fieldComponents = createLanternaComponents();

			fieldComponents.forEach((entry, component) -> {
				if (entry instanceof DialogParagraph) {
					Label paragraphLabel = (Label) component;
					paragraphLabel.setLayoutData(LANTERNA_LAYOUT_SPAN_BEGINNING);
					contentPanel.addComponent(paragraphLabel);
				} else {
					contentPanel.addComponent(new Label(entry.label()));
					contentPanel.addComponent(fieldComponents.get(entry));
				}
			});

			Button submitButton = new Button("Submit", window::close);
			Panel buttonPanel = new Panel(new GridLayout(2));
			buttonPanel.addComponent(new EmptySpace(new TerminalSize(0, 0)));
			buttonPanel.addComponent(submitButton);
			buttonPanel.setLayoutData(LANTERNA_LAYOUT_SPAN_END);
			contentPanel.addComponent(buttonPanel);

			Panel mainPanel = new Panel();
			mainPanel.setLayoutManager(new LinearLayout(Direction.VERTICAL));
			mainPanel.addComponent(contentPanel);

			window.setComponent(mainPanel);

			VirtualScreen virtualScreen = new VirtualScreen(screen);

			MultiWindowTextGUI gui = new MultiWindowTextGUI(
					virtualScreen,
					new DefaultWindowManager(),
					new EmptySpace()
			);

			gui.addWindowAndWait(window);

			SequencedMap<DialogEntry, String> results = new LinkedHashMap<>();
			for (DialogEntry entry : entries) {
				Component component = fieldComponents.get(entry);
				switch (component) {
					case TextBox textBox -> results.put(entry, textBox.getText());
					case CheckBox checkBox -> results.put(entry, Boolean.toString(checkBox.isChecked()));
					case RadioBoxList<?> radioBoxList -> {
						int selectedIndex = radioBoxList.getSelectedIndex();
						List<String> options = ((DialogRadioSelect) entry).options();
						results.put(entry, selectedIndex >= 0 && selectedIndex < options.size()
								? options.get(selectedIndex) : "");
					}
					case Label ignored -> results.put(entry, "");
					default -> throw new IllegalStateException("not supported component type: " + component.getClass());
				}
			}

			screen.stopScreen();
			terminal.close();

			return results;
		} catch (IOException e) {
			throw new RuntimeException("Failed to show Lanterna dialog", e);
		}
	}

	private Map<DialogEntry, Component> createLanternaComponents() {
		Map<DialogEntry, Component> fieldComponents = new LinkedHashMap<>();

		for (DialogEntry entry : entries) {
			switch (entry) {
				case DialogParagraph paragraph -> {
					Label label = new Label(paragraph.label().strip());
					fieldComponents.put(entry, label);
				}
				case DialogTextField textField -> {
					TextBox textBox = new TextBox(new TerminalSize(textField.columns(), textField.rows()));
					if (textField.isPassword()) {
						textBox.setMask('*');
					}
					textBox.setText(textField.defaultValue());
					fieldComponents.put(entry, textBox);
				}
				case DialogCheckBox checkBox -> {
					CheckBox checkComponent = new CheckBox("");
					checkComponent.setChecked(checkBox.defaultValue());
					fieldComponents.put(entry, checkComponent);
				}
				case DialogRadioSelect radioButton -> {
					RadioBoxList<String> radioBoxList = new RadioBoxList<>();
					radioButton.options().forEach(radioBoxList::addItem);
					radioBoxList.setCheckedItemIndex(radioButton.defaultSelectedIndex());
					fieldComponents.put(entry, radioBoxList);
				}
			}
		}
		return fieldComponents;
	}

	public static void showMessage(String title, String message) {
		Dialog dialog = new Dialog(title);
		dialog.add(new DialogParagraph(message));
		dialog.show();
	}

	public static String readLine(String prompt) {
		return readUserInput(prompt, textField(""), null);
	}

	public static String readMultilineText(String prompt) {
		return readUserInput(prompt, textArea(""), null);
	}

	public static String readPassword(String prompt) {
		return readUserInput(prompt, passwordField(""), null);
	}

	public static String readMultilinePassword(String prompt) {
		return readUserInput(prompt, passwordArea(""), null);
	}

	public static String readLine(String prompt, UiMode mode) {
		return readUserInput(prompt, textField(""), mode);
	}

	public static String readPassword(String prompt, UiMode mode) {
		return readUserInput(prompt, passwordField(""), mode);
	}

	public static String readMultilinePassword(String prompt, UiMode mode) {
		return readUserInput(prompt, passwordArea(""), mode);
	}

	private static String readUserInput(String prompt, DialogEntry entry, UiMode mode) {
		Dialog dialog = new Dialog(prompt);
		dialog.add(entry);

		if (mode != null) {
			dialog.setUiMode(mode);
		}

		Map<DialogEntry, String> result = dialog.show();
		return result.get(entry);
	}


	public static DialogTextField textField(String label) {
		return new DialogTextField(label, "", 1, 50, false);
	}

	public static DialogTextField passwordField(String label) {
		return new DialogTextField(label, "", 1, 50, true);
	}

	public static DialogTextField textArea(String label) {
		return new DialogTextField(label, "", 15, 72, false);
	}

	public static DialogTextField passwordArea(String label) {
		return new DialogTextField(label, "", 15, 72, true);
	}

	public static DialogCheckBox checkBox(String label, boolean defaultValue) {
		return new DialogCheckBox(label, defaultValue);
	}

	public static DialogRadioSelect radioSelect(String label, List<String> options) {
		return new DialogRadioSelect(label, options);
	}

	public static DialogRadioSelect radioSelect(String label, List<String> options, int defaultSelectedIndex) {
		return new DialogRadioSelect(label, options, defaultSelectedIndex);
	}

	public static DialogParagraph paragraph(String text) {
		return new DialogParagraph(text);
	}


	public enum UiMode {
		ZENITY,
		LANTERNA
	}

	public sealed interface DialogEntry permits DialogCheckBox, DialogParagraph, DialogRadioSelect, DialogTextField {
		String label();
	}

	public record DialogTextField(
			String label,
			String defaultValue,
			int rows,
			int columns,
			boolean isPassword
	) implements DialogEntry {
	}

	public record DialogCheckBox(String label, boolean defaultValue) implements DialogEntry {
	}

	public record DialogRadioSelect(
			String label,
			List<String> options,
			int defaultSelectedIndex
	) implements DialogEntry {
		public DialogRadioSelect {
			if (options.isEmpty())
				throw new IllegalArgumentException("Options list cannot be empty");
			if (defaultSelectedIndex < 0 || defaultSelectedIndex >= options.size())
				throw new IllegalArgumentException("Default selected index is out of bounds");
		}

		public DialogRadioSelect(String label, List<String> options) {
			this(label, options, 0);
		}
	}

	public record DialogParagraph(
			String label
	) implements DialogEntry {
	}

}
