package fi.evolver.script;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public final class FileUtil {

	private FileUtil() {
	}

	public static void write(Path path, String content) {
		try {
			Files.writeString(path, content, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
		} catch (IOException e) {
			throw new RuntimeException("Failed to write to file " + path, e);
		}
	}

	public static void write(Path path, String content, Set<PosixFilePermission> permissions) {
		try {
			if (Files.exists(path)) {
				Files.setPosixFilePermissions(path, permissions);
			} else {
				Files.createFile(path, PosixFilePermissions.asFileAttribute(permissions));
			}
			write(path, content);
		} catch (IOException e) {
			throw new RuntimeException("Failed to write to file " + path, e);
		}
	}

	public static void append(Path path, String content) {
		try {
			Files.writeString(path, content, StandardOpenOption.CREATE, StandardOpenOption.APPEND);
		} catch (IOException e) {
			throw new RuntimeException("Failed to append to file " + path, e);
		}
	}

	public static void writeBytes(Path file, byte[] content) {
		try {
			Files.write(file, content);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static Path createTempDir(String prefix) {
		try {
			Path tempFile = Files.createTempDirectory(prefix);
			tempFile.toFile().deleteOnExit();
			return tempFile;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static Path createTempFile(String prefix, String suffix) {
		try {
			Path tempFile = Files.createTempFile(prefix, suffix);
			tempFile.toFile().deleteOnExit();
			return tempFile;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static Path writeTempFile(String prefix, String suffix, String data) {
		Path tempFile = createTempFile(prefix, suffix);
		write(tempFile, data);
		return tempFile;
	}

	public static Path writeTempFile(String prefix, String suffix, byte[] data) {
		Path tempFile = createTempFile(prefix, suffix);
		writeBytes(tempFile, data);
		return tempFile;
	}

	public static void sudoWrite(String targetPath, String content, String owner, String chmodPerms) {
		Path temp = writeTempFile("temp", ".tmp", content);
		sudoCopy(temp.toString(), targetPath, owner, chmodPerms);
	}

	public static void sudoWrite(String targetPath, byte[] content, String owner, String chmodPerms) {
		Path temp = writeTempFile("temp", ".tmp", content);
		sudoCopy(temp.toString(), targetPath, owner, chmodPerms);
	}

	public static void sudoCopy(String sourcePath, String targetPath, String owner, String chmodPerms) {
		Shell.sudo("cp", sourcePath, targetPath);
		Shell.sudo("chown", owner, targetPath);
		Shell.sudo("chmod", chmodPerms, targetPath);
	}

	public static void copy(Path source, Path target) {
		try {
			Files.createDirectories(target.getParent());
			Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING);
		} catch (IOException e) {
			throw new RuntimeException("Failed to copy file from %s to %s".formatted(source, target), e);
		}
	}

	public static void copy(Path source, Path target, Set<PosixFilePermission> permissions) {
		try {
			Files.createDirectories(target.getParent());
			Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING);
			Files.setPosixFilePermissions(target, permissions);
		} catch (IOException e) {
			throw new RuntimeException("Failed to copy file from %s to %s".formatted(source, target), e);
		}
	}

	/**
	 * @see #addShellBlock(String, String, String)
	 */
	public static void addShellBlock(Path file, String blockId, String blockContent) {
		if (blockContent.endsWith("\n")) {
			blockContent = blockContent.substring(0, blockContent.length() - 1);
		}

		try {
			String originalContent = Files.exists(file) ? Files.readString(file) : "";

			String updatedContent = addShellBlock(originalContent, blockId, blockContent);

			Files.writeString(file, updatedContent, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
		} catch (IOException e) {
			throw new RuntimeException("Failed to write to shell script file: " + file, e);
		}
	}

	/**
	 * <p>Writes a script block, replacing existing block contents if found.
	 * The block is delimited with comment strings using the blockId param.</p>
	 * E.g.
	 * <pre>
	 *     #BLOCK-START this-is-a-block-id #####
	 *     export PATH="$PATH:/new-dir"
	 *     #BLOCK-END this-is-a-block-id #####
	 * </pre>
	 *
	 * @param file a shell script file such as ~/.bashrc
	 * @param blockId the identifier used for the block
	 * @param blockContent content to write between block start and end markers
	 */
	public static String addShellBlock(String originalContent, String blockId, String blockContent) {
		String startMarker = "#BLOCK-START " + blockId + " #####";
		String endMarker = "#BLOCK-END " + blockId + " #####";

		List<String> updatedLines = new ArrayList<>();
		boolean inBlock = false;
		boolean blockFound = false;

		for (String line : originalContent.split("\n")) {
			if (line.trim().equals(startMarker)) {
				inBlock = true;
				updatedLines.add(startMarker);
				updatedLines.add(blockContent);
				blockFound = true;
				continue;
			}
			if (line.trim().equals(endMarker)) {
				inBlock = false;
				updatedLines.add(endMarker);
				continue;
			}
			if (!inBlock) {
				updatedLines.add(line);
			}
		}

		if (inBlock) {
			throw new IllegalStateException("Block " + blockId + " started but not finished");
		}

		if (!blockFound) {
			updatedLines.add("");
			updatedLines.add(startMarker);
			updatedLines.add(blockContent);
			updatedLines.add(endMarker);
			updatedLines.add("");
		}
		return String.join("\n", updatedLines);
	}

}
