package fi.evolver.script.app;

import java.io.ByteArrayInputStream;
import java.net.URI;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.stream.Stream;

import fi.evolver.script.Download;
import fi.evolver.script.FileUtil;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

import static java.util.function.Predicate.not;

public class Apt {

	/**
	 * Install the specified packages with APT.
	 *
	 * @param packageNames Packages to install.
	 */
	public static void install(String... packageNames) {
		try (Step step = Step.start("Install %s".formatted(String.join(", ", packageNames)))) {
			Shell.sudo(Stream.concat(Stream.of("apt-get", "install", "-y"), Arrays.stream(packageNames)).toList());
		}
	}

	public static void installFromUrl(String url) {
		try (Step step = Step.start("Install deb from %s".formatted(url))) {
			Path debPath = Download.intoTempFile(url);
			install(debPath.toString());
		}
	}

	public static void purge(String... packageNames) {
		try (Step step = Step.start("Purge %s".formatted(String.join(", ", packageNames)))) {
			Shell.sudo(Stream.concat(Stream.of("apt-get", "purge", "-y"), Arrays.stream(packageNames)).toList());
		}
	}

	/**
	 * Update local package information cache.
	 */
	public static void update() {
		try (Step step = Step.start("Update package lists")) {
			Shell.sudo("apt-get", "update");
		}
	}

	public static boolean isInstalled(String packageName) {
		return Shell.Command.user("dpkg-query", "-W", packageName)
				.failOnError(false)
				.run()
				.success();
	}

	/**
	 * Add a public key for APT.
	 * <p>
	 * Downloads the key in binary format and passes it to gpg for dearmoring.
	 * </p>
	 *
	 * @param keyringName The name of the keyring file.
	 * @param url         URL pointing to the public key.
	 */
	public static void addPublicKey(String keyringName, URI url) {
		ensureInstalled("gpg");

		byte[] pubKeyBytes = Download.contentBytes(url);

		byte[] dearmoredPubkey = Shell.Command.user("gpg", "--dearmor")
				.stdin(new ByteArrayInputStream(pubKeyBytes))
				.binaryStdout(true)
				.run()
				.binaryStdout();

		FileUtil.sudoWrite(
				"/etc/apt/keyrings/%s".formatted(keyringName),
				dearmoredPubkey,
				"root:root",
				"644"
		);
	}

	public static void ensureInstalled(String... packageNames) {
		String[] notInstalled = Arrays.stream(packageNames)
				.filter(not(Apt::isInstalled))
				.toArray(String[]::new);

		if (notInstalled.length > 0) {
			update();
			install(notInstalled);
		}
	}

	public static void addSource(String sourceIdentifier, String sourceFileContent) {
		FileUtil.sudoWrite(
				"/etc/apt/sources.list.d/%s.list".formatted(sourceIdentifier), sourceFileContent,
				"root:root",
				"644"
		);
	}

	public static String getArchitecture() {
		return Shell.user("dpkg", "--print-architecture").trim();
	}

}
