package fi.evolver.script.app;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import fi.evolver.script.system.Architecture;
import fi.evolver.script.Download;
import fi.evolver.script.FileUtil;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class AwsCli {

	/**
	 * <b>Installs the latest AWS CLI v2.</b>
	 * <p>Downloads the appropriate archive for the current architecture,
	 * extracts and installs it, and configures bash completion.</p>
	 */
	public static void install() throws IOException {
		try (Step step = Step.start("AWS CLI: install")) {
			if (Files.exists(Path.of("/usr/local/aws-cli"))) {
				step.skip("Already installed");
				return;
			}

			Apt.install("unzip");

			String url = switch (Architecture.validated()) {
				case AMD64 -> "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip";
				case ARM64 -> "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip";
			};

			Path zipFile = Download.intoTempFile(url);
			Path tmpDir = FileUtil.createTempDir("awscli");
			Shell.user("unzip", "-o", zipFile.toString(), "-d", tmpDir.toString());

			Path installer = tmpDir.resolve("aws").resolve("install");
			Shell.sudo(installer.toString(), "--update");

			FileUtil.addShellBlock(Shell.BASHRC, "AWS CLI bash completion", """
					complete -C '/usr/local/bin/aws_completer' aws
					""");
		}
	}

	public static void installSessionManagerPlugin() throws IOException {
		try (Step step = Step.start("AWS CLI: install session manager plugin")) {
			if (Apt.isInstalled("session-manager-plugin")) {
				step.skip("Already installed");
				return;
			}

			Apt.installFromUrl(switch (Architecture.validated()) {
				case AMD64 -> "https://s3.amazonaws.com/session-manager-downloads/plugin/latest/ubuntu_64bit/session-manager-plugin.deb";
				case ARM64 -> "https://s3.amazonaws.com/session-manager-downloads/plugin/latest/ubuntu_arm64/session-manager-plugin.deb";
			});
		}
	}

}
