package fi.evolver.script.app;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import fi.evolver.script.FileUtil;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

/**
 * Note that there is no ARM Linux version of Chrome available.
 * Chromium is available e.g. through Debian repositories.
 */
public class Chrome {

	private static final String EXTERNAL_EXTENSION_JSON = """
			{
				"external_update_url": "https://clients2.google.com/service/update2/crx"
			}
			""";
	private static final Path CONFIG_DIR_CHROME = Shell.HOME.resolve(".config/google-chrome");
	private static final Path CONFIG_DIR_CHROMIUM = Shell.HOME.resolve(".config/chromium");

	public static final String EXTENSION_REACT_DEVELOPER_TOOLS = "fmkadmapgofadopljbjfkapdkoienihi";
	public static final String EXTENSION_REDUX_DEV_TOOLS = "lmhkpmbekcpmknklioeibfkpmmfibljd";

	public static void addExtensions(Flavor flavor, List<String> extensionIds) throws IOException {
		try (Step step = Step.start("%s: Add extensions".formatted(flavor.label))) {
			Path externalExtensionsDir = getConfigDir(flavor).resolve("External Extensions");
			if (Files.exists(externalExtensionsDir)) {
				step.skip("External extensions already configured");
				return;
			}

			Files.createDirectories(externalExtensionsDir);

			for (String extensionId : extensionIds) {
				Path extensionFile = externalExtensionsDir.resolve(extensionId + ".json");
				FileUtil.write(extensionFile, EXTERNAL_EXTENSION_JSON);
			}
		}
	}

	public static void installBookmarks(Flavor flavor, Path bookmarksSourcePath) throws IOException {
		try (Step step = Step.start("%s: Install Bookmarks".formatted(flavor.label))) {
			Path defaultProfileDir = getConfigDir(flavor).resolve("Default");
			Path targetFile = defaultProfileDir.resolve("Bookmarks");

			if (Files.exists(targetFile)) {
				step.skip("Bookmarks already exist");
				return;
			}

			Files.createDirectories(defaultProfileDir);
			Files.copy(
					bookmarksSourcePath,
					targetFile
			);
		}
	}

	private static Path getConfigDir(Flavor flavor) {
		return switch (flavor) {
			case CHROME -> CONFIG_DIR_CHROME;
			case CHROMIUM -> CONFIG_DIR_CHROMIUM;
		};
	}

	public enum Flavor {
		CHROME("Chrome"),
		CHROMIUM("Chromium");

		private final String label;

		Flavor(String label) {
			this.label = label;
		}

		public String getLabel() {
			return label;
		}
	}

}

