package fi.evolver.script.app;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import fi.evolver.script.Download;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

/**
 * Install IDEA and custom plugins for it from JetBrains Marketplace.
 */
public class IDEA {

	// https://plugins.jetbrains.com/plugin/18504-grails
	public static final String PLUGIN_GRAILS = "org.intellij.grails";

	// https://plugins.jetbrains.com/plugin/9333-makefile-language
	public static final String PLUGIN_MAKEFILE = "name.kropp.intellij.makefile";

	// https://plugins.jetbrains.com/plugin/4441-jrebel-and-xrebel
	public static final String PLUGIN_JREBEL = "JRebelPlugin";

	// https://plugins.jetbrains.com/plugin/9525--env-files
	public static final String PLUGIN_DOTENV = "ru.adelf.idea.dotenv";

	// https://plugins.jetbrains.com/plugin/7861-envfile
	public static final String PLUGIN_ENV_FILE = "net.ashald.envfile";

	// https://plugins.jetbrains.com/plugin/24765-multi-project-workspace
	public static final String PLUGIN_MULTIPROJECT_WORKSPACES = "org.jetbrains.plugins.workspace";

	private static final Path APPS_DIR = Shell.HOME.resolve("apps");
	private static final Path INSTALL_DIR = APPS_DIR.resolve("IntelliJIdeaUltimate");

	/**
	 * Installs IntelliJ Idea Ultimate and its plugins.
	 *
	 * <p>If IntelliJ Idea Ultimate is already installed in the target directory, the installation is skipped.
	 * Otherwise, the distribution archive is downloaded, extracted, and the resulting folder is renamed to a
	 * standardized installation folder.</p>
	 * @param ideaUrl note that there are different download urls for AMD64 and ARM64.
	 */
	public static void install(String ideaUrl) throws IOException {
		try (Step step = Step.start("IntelliJ Idea: install")) {
			if (Files.exists(INSTALL_DIR)) {
				step.skip("IntelliJ Idea Ultimate is already installed");
				return;
			}
			Files.createDirectories(APPS_DIR);

			Path downloadedTarGz = Download.intoTempFile(ideaUrl);

			Shell.user("tar", "-xzf", downloadedTarGz.toString(), "-C", APPS_DIR.toString());

			try (Stream<Path> dirStream = Files.list(APPS_DIR)) {
				Path extractedDir = dirStream
						.filter(p -> Files.isDirectory(p) && p.getFileName().toString().startsWith("idea-IU"))
						.findFirst()
						.orElseThrow(() -> new RuntimeException("Extracted IntelliJ Idea folder not found"));

				Files.move(extractedDir, INSTALL_DIR);
			}
		}
	}

	/**
	 * @see <a href="https://www.jetbrains.com/help/idea/install-plugins-from-the-command-line.html#linux">official doc</a>
	 */
	public static void installPublicPlugins(List<String> pluginIds) {
		try (Step step = Step.start("Install IDEA plugins")) {
			Path ideaSh = INSTALL_DIR.resolve("bin/idea.sh");
			if (!Files.exists(ideaSh)) {
				step.fail("idea.sh not found in " + ideaSh);
			}
			var tokens = new ArrayList<String>();
			tokens.add(ideaSh.toString());
			tokens.add("installPlugins");
			tokens.addAll(pluginIds);
			Shell.user(tokens);
		}
	}
}
