package fi.evolver.script.demo;

import java.util.List;
import java.util.Map;
import java.util.Scanner;

import fi.evolver.script.Dialog;
import fi.evolver.script.Dialog.DialogEntry;

public class DialogDemo {
	public static void main(String[] args) {
		System.out.println("Dialog Demo Application");
		System.out.println("======================");

		Scanner scanner = new Scanner(System.in);

		while (true) {
			System.out.println("\nChoose a UI mode:");
			System.out.println("1. Auto-detect");
			System.out.println("2. Force Zenity");
			System.out.println("3. Force Lanterna");
			System.out.println("0. Exit");

			System.out.print("Enter choice: ");
			String choice = scanner.nextLine();

			if (choice.equals("0")) {
				break;
			}

			Dialog.UiMode mode = switch (choice) {
				case "2" -> Dialog.UiMode.ZENITY;
				case "3" -> Dialog.UiMode.LANTERNA;
				default -> null; // Auto-detect
			};

			System.out.println("\nChoose a test:");
			System.out.println("1. Simple form dialog");
			System.out.println("2. Complex form with multiple field types");
			System.out.println("3. Test readLine() from Input replacement");
			System.out.println("4. Test readPassword() from Input replacement");
			System.out.println("5. Test readMultilinePassword() from Input replacement");
			System.out.println("6. Test Paragraph Dialog"); // Add this line

			System.out.print("Enter choice: ");
			String testChoice = scanner.nextLine();

			try {
				switch (testChoice) {
					case "1" -> testSimpleForm(mode);
					case "2" -> testComplexForm(mode);
					case "3" -> testReadLine(mode);
					case "4" -> testReadPassword(mode);
					case "5" -> testReadMultilinePassword(mode);
					case "6" -> testLabels(mode); // Add this line
					default -> System.out.println("Invalid choice");
				}
			} catch (Exception e) {
				System.err.println("Error during test: " + e.getMessage());
				e.printStackTrace(System.err);
			}		}

		scanner.close();
		System.out.println("Demo ended.");
	}

	private static void testSimpleForm(Dialog.UiMode mode) {
		Dialog dialog = new Dialog("Simple Form Test");
		if (mode != null) {
			dialog.setUiMode(mode);
		}

		dialog.add(Dialog.textField("Name"));
		dialog.add(Dialog.textField("Email"));

		System.out.println("\nShowing simple form dialog...");
		Map<DialogEntry, String> results = dialog.show();

		System.out.println("\nResults:");
		results.forEach((entry, value) ->
				System.out.println(entry.label() + ": " + value));
	}

	private static void testComplexForm(Dialog.UiMode mode) {
		Dialog dialog = new Dialog("Complex Form Test");
		if (mode != null) {
			dialog.setUiMode(mode);
		}

		dialog.add(Dialog.textField("Username"));
		dialog.add(Dialog.passwordField("Password"));
		dialog.add(Dialog.textArea("Bio"));
		dialog.add(Dialog.passwordArea("Secret Notes"));

		dialog.add(Dialog.checkBox("Subscribe to newsletter", true));

		List<String> accountTypes = List.of("Free", "Basic", "Premium", "Enterprise");
		dialog.add(Dialog.radioSelect("Account type", accountTypes, 1));

		System.out.println("\nShowing complex form dialog...");
		Map<DialogEntry, String> results = dialog.show();

		System.out.println("\nResults:");
		for (Map.Entry<DialogEntry, String> entry : results.entrySet()) {
			String name = entry.getKey().label();
			String value = entry.getValue();

			switch (entry.getKey()) {
				case Dialog.DialogTextField field when field.isPassword() ->
						System.out.println(name + ": " + "*".repeat(value.length()));
				case Dialog.DialogCheckBox ignored ->
						System.out.println(name + ": " + (value.equals("true") ? "✓ Checked" : "□ Unchecked"));
				case Dialog.DialogRadioSelect ignored ->
						System.out.println(name + ": Selected option: " + value);
				case null, default -> System.out.println(name + ": " + value);
			}
		}
	}

	private static void testReadLine(Dialog.UiMode mode) {
		System.out.println("\nTesting readLine()...");
		String input = Dialog.readLine("Enter some text:", mode);
		System.out.println("You entered: " + input);
	}

	private static void testReadPassword(Dialog.UiMode mode) {
		System.out.println("\nTesting readPassword()...");
		String password = Dialog.readPassword("Enter a password:", mode);
		System.out.println("Password length: " + password.length());
		System.out.println("Password (masked): " + "*".repeat(password.length()));
	}

	private static void testReadMultilinePassword(Dialog.UiMode mode) {
		System.out.println("\nTesting readMultilinePassword()...");
		String multilinePassword = Dialog.readMultilinePassword(
				"Enter a multiline password (like a SSH key):", mode);

		System.out.println("Multiline password length: " + multilinePassword.length());
		System.out.println("Number of lines: " + multilinePassword.lines().count());
		System.out.println("First few characters (masked): " +
				"*".repeat(Math.min(multilinePassword.length(), 10)) +
				(multilinePassword.length() > 10 ? "..." : ""));
	}

	private static void testLabels(Dialog.UiMode mode) {
		Dialog dialog = new Dialog("Information Dialog");
		if (mode != null) {
			dialog.setUiMode(mode);
		}

		dialog.add(Dialog.paragraph("This is an informational message that doesn't require any input."));
		dialog.add(Dialog.paragraph("""
				Important Information
				
				You are arrested!"""));
		dialog.add(Dialog.textField("Any comments?"));
		dialog.add(Dialog.paragraph("System Status: All systems are operational. No issues detected."));

		System.out.println("\nShowing paragraph dialog...");
		Map<DialogEntry, String> results = dialog.show();

		System.out.println("\nResults:");
		String comment = results.values().stream()
				.filter(s -> !s.isEmpty())
				.findFirst().orElse("No comments provided.");
		System.out.println("Comment: " + comment);
	}
}
