package fi.evolver.script.app;

import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import fi.evolver.script.Download;
import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Firefox {

	public static final Extension EXTENSION_MULTI_ACCOUNT_CONTAINERS = new Extension(
			"@testpilot-containers", // https://addons.mozilla.org/en-US/firefox/addon/multi-account-containers/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4355970/multi_account_containers-8.2.0.xpi"));

	public static final Extension EXTENSION_GRANTED_CONTAINERS = new Extension(
			"{b5e0e8de-ebfe-4306-9528-bcc18241a490}", // https://addons.mozilla.org/en-US/firefox/addon/granted/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4086622/granted-1.1.2.xpi"));

	private static final String KEYRING_NAME = "packages.mozilla.org.gpg";

	public static void installOfficial() {
		try (Step step = Step.start("Firefox: Install")) {
			Apt.addPublicKey(KEYRING_NAME, URI.create("https://packages.mozilla.org/apt/repo-signing-key.gpg"));
			Apt.addSource("mozilla", "deb [signed-by=/etc/apt/keyrings/%s] https://packages.mozilla.org/apt mozilla main"
					.formatted(KEYRING_NAME)
			);
			Apt.update();
			Apt.install("firefox");
		}
	}

	/**
	 * Downloads and a Firefox extension and adds it to the system autoinstall directory.
	 * This will install the extension automatically to new profiles.
	 *
	 * @param firefoxInstallDir the directory of the system Firefox installation
	 */
	public static void addDefaultExtensions(Path firefoxInstallDir, List<Extension> extensions) {
		try (Step step = Step.start("Firefox: install extensions")) {
			Path externalExtensionsDir = firefoxInstallDir.resolve("distribution/extensions");

			Shell.sudo("mkdir", "-p", externalExtensionsDir.toString());

			for (Extension extension : extensions) {
				Path downloadPath = Download.intoTempFile(extension.downloadUrl);
				Path targetPath = externalExtensionsDir.resolve(extension.id + ".xpi");
				if (Files.exists(targetPath)) {
					continue;
				}
				FileUtils.sudoCopy(downloadPath.toString(), targetPath.toString(), "root:root", "644");
			}
		}
	}

	/**
	 * @param id          the identifier of the extension (for example, "@testpilot-containers") This can be seen e.g.
	 *                    by inspecting the DOM for the Firefox about:extensions listing.
	 * @param downloadUrl the URL to download the extension .xpi file from
	 */
	public record Extension(String id, URI downloadUrl) {
	}
}
