package fi.evolver.script;

import java.time.Duration;
import java.time.Instant;

public class Step implements AutoCloseable {

	public static final String NESTING_INDENT = "--->";

	/**
	 * TODO add levels or multiple flags
	 */
	public static final boolean DEBUG_ENABLED = System.getenv().containsKey("EVOLVER_SCRIPT_DEBUG_ENABLED");

	private static Step head;

	private final Step parent;
	private final String name;
	private final Instant start;
	private final int nestingLevel;

	private String skipReason = null;

	private Step(Step parent, String name) {
		this.parent = parent;
		this.name = name;
		this.start = Instant.now();
		this.nestingLevel = parent != null ? parent.nestingLevel + 1 : 0;
		if (DEBUG_ENABLED) {
			printIndented("START %-58s%n", name);
		} else if (parent == null) {
			System.out.printf("%-58s", name);
		}
	}


	public static Step start(String name) {
		head = new Step(head, name);
		return head;
	}


	@Override
	public void close() {
		head = parent;
		if (DEBUG_ENABLED) {
			if (skipReason != null)
				printIndented("[SKIPPED: %s]%n", skipReason);
		} else if (parent == null) {
			if (skipReason != null)
				System.out.printf("SKIPPED: %s%n", skipReason);
			else
				System.out.printf("DONE in %s%n", formatDuration(Duration.between(start, Instant.now())));
		}
	}

	private String formatDuration(Duration dur) {
		long seconds = dur.toSeconds();

		if (seconds < 60)
			return "%d seconds".formatted(seconds);

		return "%d min %d seconds".formatted(seconds / 60, seconds % 60);
	}


	public void skip(String skipReason) {
		this.skipReason = skipReason;
	}


	public void fail(String reason, Throwable cause) {
		if (DEBUG_ENABLED) {
			printIndented("FAILED %-58s%n", name);
			printIndented(" - %s%n", reason);
		} else {
			System.out.printf("FAILED %n%s%n", reason);
		}
		throw new RuntimeException(cause);
	}


	public void fail(String reason) {
		fail(reason, new IllegalStateException());
	}


	public void fail(Throwable cause) {
		fail("–", cause);
	}



	private void printIndented(String format, Object... args) {
		System.err.print(NESTING_INDENT.repeat(nestingLevel));
		System.err.printf(format, args);
	}

	public void info(String text) {
		if (DEBUG_ENABLED) {
			printIndented("%s%n", text);
		}
	}
}
