package id.devcode.openapi;

import id.devcode.openapi.factory.ActuatorFactoryBean;
import id.devcode.openapi.factory.ComponentFactoryBean;
import id.devcode.openapi.factory.ExtensionFactoryBean;
import id.devcode.openapi.factory.OpenApiFactoryBean;
import static id.devcode.openapi.utils.Constant.SPRINGDOC_GROUPS_ENABLED;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.parameters.Parameter;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import static org.springdoc.core.utils.Constants.SPRINGDOC_ENABLED;
import static org.springdoc.core.utils.Constants.SPRINGDOC_SHOW_ACTUATOR;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.PropertySource;

/**
 * OpenApi auto configuration
 * @author laminalfalah on 01/09/25
 * @version 1.0.0
 */
@Slf4j
@AutoConfiguration
@EnableConfigurationProperties(OpenApiProperties.class)
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@ConditionalOnProperty(name = SPRINGDOC_ENABLED, matchIfMissing = true)
@ConditionalOnClass({
        Components.class,
        Parameter.class,
        SecurityScheme.class
})
@PropertySource("classpath:/springdoc-openapi.properties")
public class OpenApiAutoConfiguration {

    @Bean
    public ComponentFactoryBean components(ApplicationContext context) {
        Map<String, Parameter> parameters = context.getBeansOfType(Parameter.class);
        Map<String, SecurityScheme> securityScheme = context.getBeansOfType(SecurityScheme.class);
        log.debug("Found {} parameters", parameters.size());
        log.debug("Found {} security scheme", securityScheme.size());
        ComponentFactoryBean componentFactoryBean = new ComponentFactoryBean();
        componentFactoryBean.setParameters(parameters);
        componentFactoryBean.setSecuritySchemes(securityScheme);
        return componentFactoryBean;
    }

    @Bean
    public OpenApiFactoryBean openApiFactoryBean(Components components, OpenApiProperties properties) {
        log.debug("Initializing open api factory bean");
        OpenApiFactoryBean openApiFactoryBean = new OpenApiFactoryBean();
        openApiFactoryBean.setComponents(components);
        openApiFactoryBean.setOpenApiProperties(properties);
        return openApiFactoryBean;
    }

    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnClass(OpenApiProperties.class)
    @ConditionalOnProperty(name = SPRINGDOC_GROUPS_ENABLED, matchIfMissing = true)
    public ExtensionFactoryBean extensionFactoryBean(Components components, OpenApiProperties properties, @Value("${management.endpoints.web.base-path:/actuator}") String actuator) {
        log.debug("Initializing extension factory bean");
        ExtensionFactoryBean extensionFactoryBean = new ExtensionFactoryBean();
        extensionFactoryBean.setComponents(components);
        extensionFactoryBean.setOpenApiProperties(properties);
        extensionFactoryBean.setExtensionPath(actuator);
        return extensionFactoryBean;
    }

    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnClass(name = "org.springframework.boot.actuate.autoconfigure.endpoint.web.WebEndpointAutoConfiguration")
    @ConditionalOnProperty(name = {SPRINGDOC_SHOW_ACTUATOR, SPRINGDOC_GROUPS_ENABLED}, matchIfMissing = true)
    public ActuatorFactoryBean actuatorFactoryBean(Components components, OpenApiProperties properties, @Value("${management.endpoints.web.base-path:/actuator}") String actuator) {
        log.debug("Initializing actuator factory bean");
        ActuatorFactoryBean actuatorFactoryBean = new ActuatorFactoryBean();
        actuatorFactoryBean.setComponents(components);
        actuatorFactoryBean.setOpenApiProperties(properties);
        actuatorFactoryBean.setActuatorPath(actuator);
        return actuatorFactoryBean;
    }

}
