package id.devcode.openapi.factory;

import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.parameters.Parameter;
import java.util.List;
import lombok.Setter;
import org.springdoc.core.models.GroupedOpenApi;
import org.springframework.beans.factory.FactoryBean;

/**
 * Factory bean for extension.
 * @author laminalfalah on 01/09/25
 * @version 1.0.0
 */
public class ExtensionFactoryBean extends OpenApiSupport implements FactoryBean<GroupedOpenApi> {

    @Setter
    private String extensionPath;

    /**
     * Get grouped open api for extension.
     * @return grouped open api
     * @throws Exception than can be thrown by the getObject method
     */
    @Override
    public GroupedOpenApi getObject() throws Exception {
        List<String> filterParams = List.of("page_offset", "page_size", "order_column", "order_ascending");
        return GroupedOpenApi.builder()
                .displayName("OpenApi")
                .group("openapi")
                .addOpenApiCustomizer(openApi -> {
                    openApi.setInfo(getInfoOpenApi());
                    validateServer(openApi);
                    applyAdditionalComponents(openApi);
                    if (openApi.getPaths() == null || openApi.getPaths().isEmpty()) {
                        return;
                    }
                    Components components = openApi.getComponents();
                    if (components == null || components.getParameters() == null || components.getParameters().isEmpty()) {
                        return;
                    }
                    List<Parameter> componentParams = components.getParameters().values().stream()
                            .filter(p -> p.getName() != null)
                            .filter(p -> !filterParams.contains(p.getName()))
                            .toList();

                    openApi.getPaths()
                            .values()
                            .stream()
                            .flatMap(pathItem -> pathItem.readOperations().stream())
                            .forEach(operation -> {
                                List<String> existingNames = operation.getParameters() == null
                                        ? List.of()
                                        : operation.getParameters().stream()
                                        .map(Parameter::getName)
                                        .toList();
                                componentParams.stream()
                                        .filter(p -> !existingNames.contains(p.getName()))
                                        .forEach(operation::addParametersItem);
                            });
                })
                .pathsToExclude(extensionPath + "/**")
                .build();
    }

    /**
     * Get object type.
     * @return object type
     */
    @Override
    public Class<?> getObjectType() {
        return GroupedOpenApi.class;
    }

}
