package io.github.andreyzebin.gitSql.bash;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.nio.file.Path;
import java.util.function.Function;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class Bash implements BashIO {

    public static final String WIN_BASH_PATH = "C:\\Program Files\\Git\\bin\\bash.exe";

    private final boolean isSilent;
    private final Runtime runtime;

    private final Function<Runtime, Process> terminals;


    public Bash(boolean isSilent, Runtime runtime, Function<Runtime, Process> terminals) {
        this.isSilent = isSilent;
        this.runtime = runtime;
        this.terminals = terminals;
    }

    @Override
    public void runCommand(String str, PrintStream out, PrintStream err, Path cd) {
        Process t = terminals.apply(runtime);

        StringBuffer cmd = new StringBuffer();
        try (BufferedWriter bw = t.outputWriter()) {

            cmd.append(cdCommand(cd));
            cmd.append(" && ");
            cmd.append(str);

            bw.write(cmd.toString());
            if (!isSilent) {
                log.debug(Color.YELLOW + cmd.toString() + Color.RESET);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        t.inputReader().lines().forEach(out::println);
        t.errorReader().lines().forEach(err::println);
    }

    @Override
    public void runCommand(String str, PrintWriter out, PrintWriter err, Path cd) {
        Process t = terminals.apply(runtime);

        StringBuilder cmd = new StringBuilder();

        try (
                BufferedWriter bw = t.outputWriter();
                LogWriter logWriter = new LogWriter(log::debug, true);
                PrintWriter pw = logWriter.getPW()
        ) {

            cmd.append(cdCommand(cd));
            cmd.append(System.lineSeparator());
            cmd.append(str);

            bw.write(cmd.toString());
            if (!isSilent) {
                BashCSS.bashRender(cmd).lines().forEach(pw::println);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        try {
            Thread thread = asyncRead(t.inputReader(), out);
            Thread thread1 = asyncRead(t.errorReader(), err);

            thread.join();
            thread1.join();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public static Thread asyncRead(BufferedReader t, PrintWriter out) {
        Thread thread = new Thread(
                () -> {
                    try {
                        String line;
                        while ((line = t.readLine()) != null) {
                            out.println(line);
                        }
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
        );
        thread.start();
        return thread;
    }

    private String cdCommand(Path jump) {
        return "cd " + escape(toAbsolutePosix(jump));
    }

    public static String escape(String posix) {
        return "\"" + posix + "\"";
    }

    public static String decode(String value) {
        return removeSurrounding(removeSurrounding(value, "'"), "\"");
    }

    private static String removeSurrounding(String value, String prefix) {
        if (value.startsWith(prefix) && value.endsWith(prefix)) {
            return value.substring(1, value.length() - 1);
        }
        return value;
    }

    public static String toAbsolutePosix(Path p) {
        return toPosix("/" + p.toAbsolutePath().normalize());
    }

    private static String toPosix(String p) {
        return p
                .replace("\\", "/")
                .replace(":", "");
    }

    public static String toPosix(Path p) {
        return toPosix(p.toString());
    }

    public static void append(StringBuffer buf, String line) {

        if (!buf.isEmpty()) {
            buf.append(System.lineSeparator());
        }
        buf.append(line);
    }

    public static Process runWinShell(Runtime runtime) {
        return runShell(runtime, WIN_BASH_PATH);
    }

    public static Process runShell(Runtime runtime, String winBashPath) {
        Process terminal = null;
        try {
            terminal = runtime.exec(new String[]{winBashPath});
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return terminal;
    }

    public static Process runShellForOs(Runtime r) {
        boolean isWindows = System.getProperty("os.name")
                .toLowerCase().startsWith("windows");

        if (isWindows) {
            // log.debug("Windows OS is found");
            return runWinShell(r);
        }
        // log.debug("Unix OS is found");
        return runShell(r, "/bin/sh");
    }
}
