package io.github.andreyzebin.gitSql.bash;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.function.Consumer;
import java.util.function.Function;

public class LogWriter implements AutoCloseable {

    private final Consumer<String> logger;
    private final ByteArrayOutputStream bs = new ByteArrayOutputStream();
    private final boolean splitLines;
    private final Function<String, String> afterRender;

    public LogWriter(Consumer<String> logger) {
        this(logger, false);
    }

    public LogWriter(Consumer<String> logger, boolean splitLines) {
        this(logger, splitLines, f -> f);
    }

    public LogWriter(Consumer<String> logger, boolean splitLines, Function<String, String> afterRender) {
        this.logger = logger;
        this.splitLines = splitLines;
        this.afterRender = afterRender;
    }

    public PrintStream getPS() {
        if (splitLines) {
            return new PrintStream(bs) {
                @Override
                public void println(String x) {
                    super.println(x);
                    logger.accept(x);
                }
            };
        } else {
            return new PrintStream(bs);
        }
    }

    public PrintWriter getPW() {
        if (splitLines) {
            return new PrintWriter(bs) {
                @Override
                public void println(String x) {
                    super.println(x);
                    logger.accept(afterRender.apply(x));
                }
            };
        } else {
            return new PrintWriter(bs);
        }
    }

    @Override
    public void close() throws Exception {
        if (!isBlank()) {
            if (!splitLines) {
                logger.accept(bs.toString());
            }
        }
    }

    public boolean isBlank() {
        return bs.toString().replace(System.lineSeparator(), "").isBlank();
    }

    @Override
    public String toString() {
        return bs.toString();
    }

    public static PrintWriter fork(OutputStream a, PrintWriter b) {
        return new PrintWriter(a) {
            @Override
            public void println(String x) {
                super.println(x);
                b.println(x);
            }
        };
    }
}
