package io.github.andreyzebin.gitSql.git;

import io.github.andreyzebin.gitSql.FSOffsetProxy;
import io.github.andreyzebin.gitSql.FileSystem;
import io.github.andreyzebin.gitSql.bash.Bash;
import io.github.andreyzebin.gitSql.bash.BashIO;
import io.github.andreyzebin.gitSql.bash.BashTML;
import io.github.andreyzebin.gitSql.bash.Color;
import io.github.andreyzebin.gitSql.bash.LogWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Arrays;
import java.util.Comparator;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class GitFiles implements FileSystem, VersionControl {

    private final BashIO bash;
    private final boolean isReadOnly;
    private final GitAuth authStrategy;
    private final GitSource source;
    private StringBuffer filesToAdd;

    public GitFiles(BashIO bash, boolean isReadOnly, GitAuth authStrategy, GitSource source) {
        this.bash = bash;
        this.isReadOnly = isReadOnly;
        this.authStrategy = authStrategy;
        this.source = source;
        filesToAdd = new StringBuffer();
    }

    @Override
    public Writer put(Path path) {
        try {
            Path newFile = getAbs(path);
            Files.createDirectories(newFile.getParent());
            Bash.append(filesToAdd, path.toString());

            return new FileWriter(newFile.toFile(), false);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Path getAbs(Path path) {
        return source.getRoot().resolve(path);
    }

    @Override
    public boolean erase(Path path) {
        try {
            Path newFile = getAbs(path);
            if (Files.exists(newFile)) {
                Files.deleteIfExists(newFile);
                return true;
            }
            return false;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Writer patch(Path path) {
        try {
            Path newFile = getAbs(path);
            Files.createDirectories(newFile.getParent());
            Bash.append(filesToAdd, path.toString());

            return new FileWriter(newFile.toFile(), true);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Reader get(Path path) {
        try {
            return new FileReader(getAbs(path).toFile());
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean exists(Path path) {
        return getAbs(path).toFile().exists();
    }

    @Override
    public boolean isDir(Path path) {
        return getAbs(path).toFile().isDirectory();
    }

    @Override
    public String run(String cmd, PrintStream stdOut, PrintStream stdErr, Path path) {
        bash.runCommand(cmd, stdOut, stdErr, source.getRoot().resolve(path));
        return "";
    }

    @Override
    public String run(String cmd, PrintWriter stdOut, PrintWriter stdErr, Path path) {
        bash.runCommand(cmd, stdOut, stdErr, source.getRoot().resolve(path));
        return "";
    }

    @Override
    public void find(Function<Path, Boolean> needContinue, Consumer<Path> sayBye, Comparator<Path> sorting) {
        Path root = source.getRoot();
        FileSystem.tVerse(
                root,
                (p) -> needContinue.apply(root.relativize(p)),
                (p) -> sayBye.accept(root.relativize(p)),
                sorting
        );
    }

    @Override
    public void flush() {
        if (isReadOnly) {
            return;
        }
        Path root = source.getRoot();

        if (!filesToAdd.toString().isBlank()) {
            Arrays.stream(filesToAdd.toString().split(System.lineSeparator()))
                    .forEach(cFile -> {
                        try (
                                LogWriter errLog = new LogWriter(
                                        log::error,
                                        true,
                                        f -> new BashTML(f)
                                                .fill(Color.RED)
                                                .paint("ERROR", Color.RED_BOLD_BRIGHT)
                                                .toString()
                                );
                                LogWriter outLog = new LogWriter(log::debug, true,
                                        f -> new BashTML(f).fill(Color.GREEN).toString());
                        ) {
                            GitCommands.add(root, bash, Path.of(cFile), outLog.getPW(), errLog.getPW());
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    });
        }
        try (
                LogWriter errLog = new LogWriter(log::error, true, f -> new BashTML(f)
                        .fill(Color.RED)
                        .paint("ERROR", Color.RED_BOLD_BRIGHT)
                        .toString());
                LogWriter outLog = new LogWriter(log::debug, true, f -> new BashTML(f).fill(Color.GREEN).toString());
        ) {
            GitCommands.commit(root, bash, outLog.getPW(), errLog.getPW());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        try (
                LogWriter errLog = new LogWriter(log::error, true,
                        f -> new BashTML(f)
                                .fill(Color.RED)
                                .paint("ERROR", Color.RED_BOLD_BRIGHT)
                                .toString());
                LogWriter outLog = new LogWriter(log::debug, true, f -> new BashTML(f).fill(Color.GREEN).toString());
        ) {
            GitCommands.push(root, bash, outLog.getPW(), errLog.getPW(), authStrategy.useSshAgent());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        filesToAdd = new StringBuffer();
    }

    @Override
    public FileSystem cd(Path jump) {
        return new FSOffsetProxy(this, jump);
    }

    /**
     * Fast-forward
     */
    @Override
    public GitFiles seek(String commit) {
        try (
                LogWriter errLog = new LogWriter(log::error, true, f -> new BashTML(f)
                        .fill(Color.RED)
                        .paint("ERROR", Color.RED_BOLD_BRIGHT)
                        .toString());
                LogWriter outLog = new LogWriter(log::debug, true, f -> new BashTML(f).fill(Color.GREEN).toString());
        ) {
            GitCommands.checkout(source.getRoot(), commit, bash, outLog.getPW(), errLog.getPW(),
                    authStrategy.useSshAgent());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return this;
    }

    /**
     * Fast-forward
     */
    @Override
    public Stream<Commit> commits() {
        try (LogWriter errLog = new LogWriter(log::error, true, f -> new BashTML(f)
                .fill(Color.RED)
                .paint("ERROR", Color.RED_BOLD_BRIGHT)
                .toString());
                LogWriter outLog = new LogWriter(log::debug, true, f -> new BashTML(f).fill(Color.GREEN).toString());) {
            return GitCommands.commitsList(source.getRoot(), bash, outLog.getPW(), errLog.getPW());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Instant timestamp() {
        try (LogWriter errLog = new LogWriter(log::error, true, f -> new BashTML(f)
                .fill(Color.RED)
                .paint("ERROR", Color.RED_BOLD_BRIGHT)
                .toString());
                LogWriter outLog = new LogWriter(log::debug, true, f -> new BashTML(f).fill(Color.GREEN).toString());) {
            return GitCommands.commitsList(source.getRoot(), bash, outLog.getPW(), (errLog).getPW())
                    .findFirst()
                    .get()
                    .getTimestampInstant();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws Exception {
        flush();
    }

}
