package io.github.andreyzebin.gitSql.git;

import io.github.andreyzebin.gitSql.bash.BashIO;
import io.github.andreyzebin.gitSql.bash.BashTML;
import io.github.andreyzebin.gitSql.bash.Color;
import io.github.andreyzebin.gitSql.bash.LogWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class RemoteSource implements GitSource {

    private static final String TMP_DIR_NAME = "gitfs_tmp";
    private final String gitUri;
    private final BashIO bash;
    private final Path home;
    private final GitAuth authStrategy;
    private Path root;

    public RemoteSource(String gitUri, BashIO bash, Path home, GitAuth authStrategy) {
        this.gitUri = gitUri;
        this.bash = bash;
        this.home = home;
        this.authStrategy = authStrategy;
    }

    @Override
    public Path getRoot() {
        if (root == null) {
            log.debug("Initializing Git root...");
            Path tmp = home.resolve(TMP_DIR_NAME).resolve(String.valueOf(System.nanoTime()));
            try (
                    LogWriter errLog = new LogWriter(log::error, true,
                            f -> new BashTML(f)
                                                .fill(Color.RED)
                                                .paint("ERROR", Color.RED_BOLD_BRIGHT)
                                                .toString());
                    LogWriter outLog = new LogWriter(log::debug, true,
                            f -> new BashTML(f).fill(Color.GREEN).toString());
            ) {
                Files.createDirectories(tmp.getParent());

                GitCommands.clone(
                        tmp.getFileName().toString(),
                        bash,
                        outLog.getPW(),
                        errLog.getPW(),
                        gitUri,
                        false,
                        tmp.getParent(),
                        authStrategy.getAuthHeader(),
                        authStrategy.useSshAgent()
                );

                if (errLog.toString().contains("ERROR")) {
                    throw new RuntimeException("Could not clone remote source: " + gitUri);
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            if (!tmp.toFile().exists()) {
                throw new RuntimeException("Could not clone remote source: " + gitUri);
            }
            root = tmp;
        }
        return root;
    }
}
